use sha1::{Digest, Sha1};

pub trait IaqmsHash {
    fn get(&self) -> String;
    fn compare(&self, x: Sha1) -> bool;
}

pub struct IaqmsSha1 {
    _digest: Digest,
}

impl IaqmsSha1 {
    pub fn from_json_value(value: &serde_json::Value) -> Self {
        let string = serde_json::to_string(value).unwrap();
        let mut hasher = Sha1::new();
        hasher.update(string.as_bytes());
        return IaqmsSha1 {
            _digest: hasher.digest(),
        };
    }
}

impl IaqmsHash for IaqmsSha1 {
    fn get(&self) -> String {
        return self._digest.to_string();
    }

    fn compare(&self, x: Sha1) -> bool {
        return x.digest().to_string() == self._digest.to_string();
    }
}

#[cfg(test)]
mod sha1_tests {
    use super::*;
    #[test]
    fn valid_json_returns_hex() {
        let json = "{\"device\":\"iaqms\",\"upswift\":{\"device_token\":\"\",\"user_token\":\"\"},\"version\":\"v1.0.0-alpha.0\"}";
        let sha1 = Sha1::from(json).digest().to_string();
        let hex = IaqmsSha1::from_json_value(&serde_json::from_str(json).unwrap()).get();

        assert_eq!(hex == sha1, true);
    }

    #[test]
    fn hex_compares_to_hex() {
        let json = "{\"device\":\"iaqms\",\"upswift\":{\"device_token\":\"\",\"user_token\":\"\"},\"version\":\"v1.0.0-alpha.0\"}";
        let sha1 = Sha1::from(json);
        let hex = IaqmsSha1::from_json_value(&serde_json::from_str(json).unwrap());
        assert_eq!(hex.compare(sha1), true);
        assert_eq!(
            hex.compare(Sha1::from("fcd3994ef66ac0e5674757eb371ff1e0")),
            false
        );
    }
}
