use crate::iaqms_md5::IaqmsChecksum;
use crate::iaqms_redis::IaqmsRedis;

pub const SETTING_NAME_LAST_SYNC: &str = "_last_synced";
pub const SETTING_NAME_MD5: &str = "_md5_hash";

pub struct IaqmsSettings {
    opt: IaqmsSyncOpts,
    redis: IaqmsRedis,
}

pub struct IaqmsSyncOpts {
    pub fs_sync_secs: i64,
    pub fs_setting_path: String,
    pub redis_key_name_base: String,
}

impl IaqmsSettings {
    pub fn new(redis: IaqmsRedis, opt: IaqmsSyncOpts) -> IaqmsSettings {
        return IaqmsSettings {
            opt: opt,
            redis: redis,
        };
    }

    pub fn from_file(&self, path: &str) -> serde_json::Value {
        let mut file = std::fs::read_to_string(path).expect("unable to find file");
        return serde_json::from_str(&mut file).expect("incorrectly formatted JSON");
    }
    fn _set_from_obj(&self, settings: &serde_json::Value, base: Option<&str>) {
        if !settings.is_object() {
            return;
        }

        let redis = &self.redis;
        for key in settings.as_object().unwrap().keys() {
            let object = settings.as_object().unwrap();
            if object[key].is_object() {
                self._set_from_obj(&object[key], Option::from(key.as_str()));
                continue;
            }

            let mut name: String = key.to_string();
            if base.is_some() {
                name = format!("{}_{}", base.unwrap(), key);
            }

            name = format!("{}.{}", self.opt.redis_key_name_base, name);
            let _: redis::RedisResult<()> = redis.set(&name, &object[key]);
        }
    }

    pub fn sync(&self) {
        let redis = &self.redis;
        let now = std::time::UNIX_EPOCH.elapsed().unwrap().as_secs_f64() as i64;
        let last_sync = redis.get(SETTING_NAME_LAST_SYNC).unwrap_or(-1) as i64;

        if last_sync != -1 || now - last_sync > self.opt.fs_sync_secs {
            let settings = self.from_file(&self.opt.fs_setting_path);
            let hash = IaqmsChecksum::from_json(&settings);
            let oldhash: String = redis.get(SETTING_NAME_MD5).unwrap_or(String::from(""));

            if hash.compare(oldhash) {
                return;
            }

            self._set_from_obj(&settings, None);
        }
    }
}
