use unicase::Ascii;

header! {
    /// `Access-Control-Allow-Headers` header, part of
    /// [CORS](http://www.w3.org/TR/cors/#access-control-allow-headers-response-header)
    ///
    /// The `Access-Control-Allow-Headers` header indicates, as part of the
    /// response to a preflight request, which header field names can be used
    /// during the actual request.
    ///
    /// # ABNF
    ///
    /// ```text
    /// Access-Control-Allow-Headers: "Access-Control-Allow-Headers" ":" #field-name
    /// ```
    ///
    /// # Example values
    /// * `accept-language, date`
    ///
    /// # Examples
    ///
    /// ```
    /// # extern crate http;
    /// # extern crate hyperx;
    /// # extern crate unicase;
    /// # fn main() {
    /// // extern crate unicase;
    ///
    /// use hyperx::header::{AccessControlAllowHeaders, TypedHeaders};
    /// use unicase::Ascii;
    ///
    /// let mut headers = http::HeaderMap::new();
    /// headers.encode(
    ///     &AccessControlAllowHeaders(vec![Ascii::new("date".to_owned())])
    /// );
    /// # }
    /// ```
    ///
    /// ```
    /// # extern crate http;
    /// # extern crate hyperx;
    /// # extern crate unicase;
    /// # fn main() {
    /// // extern crate unicase;
    ///
    /// use hyperx::header::{AccessControlAllowHeaders, TypedHeaders};
    /// use unicase::Ascii;
    ///
    /// let mut headers = http::HeaderMap::new();
    /// headers.encode(
    ///     &AccessControlAllowHeaders(vec![
    ///         Ascii::new("accept-language".to_owned()),
    ///         Ascii::new("date".to_owned()),
    ///     ])
    /// );
    /// # }
    /// ```
    (AccessControlAllowHeaders, "Access-Control-Allow-Headers") => (Ascii<String>)*

    test_access_control_allow_headers {
        test_header!(test1, vec![b"accept-language, date"]);
    }
}

standard_header!(AccessControlAllowHeaders, ACCESS_CONTROL_ALLOW_HEADERS);
