use std::{
    io::{Read, Seek, SeekFrom},
    num::ParseIntError,
};

use clap::{App, AppSettings, Arg};

fn try_parse_number(input: &str) -> Result<(), String> {
    match parse_number(input) {
        Ok(_) => Ok(()),
        Err(_) => Err(format!("Failed to parse '{}' as number", input)),
    }
}

fn non_zero(input: &str) -> Result<(), String> {
    match parse_number(input) {
        Ok(num) => {
            if num == 0 {
                Err(format!("'{}' must be larger than 0", input))
            } else {
                Ok(())
            }
        }
        Err(_) => Err(format!("Failed to parse '{}' as number", input)),
    }
}

fn parse_number(input: &str) -> Result<u64, ParseIntError> {
    if input.len() > 2 && input.starts_with("0x") {
        u64::from_str_radix(&input[2..], 16)
    } else if input.len() > 1 && input.starts_with('$') {
        u64::from_str_radix(&input[1..], 16)
    } else if input.len() > 1 && input.ends_with('h') {
        u64::from_str_radix(&input[..input.len() - 1], 16)
    } else if input.len() > 2 && input.starts_with("0o") {
        u64::from_str_radix(&input[2..], 8)
    } else if input.len() > 1 && input.starts_with('&') {
        u64::from_str_radix(&input[1..], 8)
    } else if input.len() > 1 && input.ends_with('o') {
        u64::from_str_radix(&input[..input.len() - 1], 8)
    } else if input.len() > 2 && input.starts_with("0b") {
        u64::from_str_radix(&input[2..], 2)
    } else if input.len() > 1 && input.starts_with('%') {
        u64::from_str_radix(&input[1..], 2)
    } else if input.len() > 1 && input.ends_with('b') {
        u64::from_str_radix(&input[..input.len() - 1], 2)
    } else {
        input.parse()
    }
}

fn layout_validator(input: &str) -> Result<(), String> {
    for c in input.chars() {
        if c != 'x' && c != 'a' && c != 'X' && c != 'c' {
            return Err(format!("Invalid layout string '{}'", input));
        }
    }
    Ok(())
}

fn main() {
    let matches = App::new("hxdmp - the no-nonsense hexdumper")
        .about(
            "Rust port by Molive.\n\
            Based on the C version by lunasorcery.",
        )
        .setting(AppSettings::DisableVersion)
        .setting(AppSettings::NextLineHelp)
        .setting(AppSettings::UnifiedHelpMessage)
        .setting(AppSettings::DeriveDisplayOrder)
        .setting(AppSettings::ColoredHelp)
        .args(&[
            Arg::new("help")
                .short('h')
                .long("help")
                .about("Display this help menu."),
            Arg::new("start")
                .short('s')
                .long("start")
                .value_name("offset")
                .about("Skip <offset> bytes from the start of the input.\n")
                .validator(try_parse_number)
                .default_value("0"),
            Arg::new("length")
                .short('n')
                .long("length")
                .value_name("length")
                .about("Only read <length> bytes of the input. Use very large number for no limit.\n")
                .validator(try_parse_number)
                .default_value("0xFFFFFFFFFFFFFFFF"),
            Arg::new("width")
                .short('w')
                .long("width")
                .value_name("width")
                .about(
                    "Display <width> bytes per row.\n\
                    You can replace the default with the HXDMP_WIDTH environment variable.\n",
                )
                .env("HXDMP_WIDTH")
                .validator(non_zero)
                .default_value("16"),
            Arg::new("layout")
                .short('l')
                .long("layout")
                .value_name("layout_str")
                .about(
                    "Customize the column layout using a layout string.\n\
                    Each character of <layout_str> represents a column.\n\
                    Available column types are:\n\
                    a (ascii) Each byte is printed in ascii, or '.' for non-ascii values\n\
                    c (color) Each byte is printed as a block colored according to its value\n\
                    x (hex)   Each byte is printed as two lowercase hex digits\n\
                    X (hex)   Each byte is printed as two uppercase hex digits\n\
                    The default layout string is xa for consistency with most hex editors.\n\
                    You can replace the default with the HXDMP_LAYOUT environment variable.\n",
                )
                .env("HXDMP_LAYOUT")
                .validator(layout_validator)
                .default_value("xa"),
            Arg::new("filepath")
                .required(true)
                .index(1)
                .hidden(true)
                .multiple(true),
        ])
        //.override_usage("hxdmp [options] <filepath>")
        .after_help("    Numeric values can be provided in hexadecimal, octal, and binary.\n    0xFF, $FF, FFh, 0o77, &77, 77o, 0b11, %11, and 11b syntaxes are all supported.")
        .get_matches();

    let width = parse_number(matches.value_of("width").expect("Error retrieving width"))
        .expect("Error converting width to number") as usize;
    let length = parse_number(matches.value_of("length").expect("Error retrieving length"))
        .expect("Error converting length to number") as usize;
    let start = parse_number(matches.value_of("start").expect("Error retrieving start"))
        .expect("Error converting start to number");
    let layout = matches.value_of("layout").expect("Error retrieving layout");
    layout_validator(layout).unwrap();

    for (i, file) in matches
        .values_of("filepath")
        .expect("Error retrieving filepath")
        .enumerate()
    {
        let mut fh = std::fs::File::open(file)
            .unwrap_or_else(|_| panic!("Failed to open '{}' for reading.", file));
        if i > 0 {
            println!();
        }
        println!("{}", file);
        fh.seek(SeekFrom::Start(start)).unwrap_or_else(|_| {
            panic!(
                "Failed to seek to start point '{}' in file '{}'",
                start, file
            )
        });

        let mut row_buffer = vec![0u8; width as usize];
        let mut total_bytes_read = 0;
        loop {
            let row_address = fh
                .stream_position()
                .unwrap_or_else(|_| panic!("Failed read position in file '{}'", file));
            let mut bytes_read = fh.read(&mut row_buffer).unwrap_or_else(|_| {
                panic!(
                    "Failed read from file '{}' as position '{}'",
                    file, row_address
                )
            });
            total_bytes_read += bytes_read;
            let should_break = if total_bytes_read > length {
                bytes_read -= total_bytes_read - length;
                true
            } else {
                false
            };
            if bytes_read == 0 {
                break;
            }
            print!("{:08x?} |", row_address);
            for char in layout.chars() {
                match char {
                    'x' => {
                        print!(" ");
                        for char in row_buffer.iter().take(width.min(bytes_read)) {
                            print!("{:02x?} ", char);
                        }
                        for _ in bytes_read..width {
                            print!("   ");
                        }
                        print!("|");
                    }
                    'X' => {
                        print!(" ");
                        for char in row_buffer.iter().take(width.min(bytes_read)) {
                            print!("{:02X?} ", char);
                        }
                        for _ in bytes_read..width {
                            print!("   ");
                        }
                        print!("|");
                    }
                    'a' => {
                        print!(" ");
                        for char in row_buffer
                            .iter()
                            .take(width.min(bytes_read))
                            .map(|c| *c as char)
                        {
                            print!(
                                "{}",
                                if char.is_ascii() && !char.is_ascii_control() {
                                    char
                                } else {
                                    '.'
                                }
                            );
                        }
                        for _ in bytes_read..width {
                            print!(" ");
                        }
                        print!(" |");
                    }
                    'c' => {
                        todo!()
                    }
                    _ => unreachable!(),
                }
            }
            println!();
            if should_break {
                break;
            }
        }
    }
}
