//! Main library entry point for hvcg_governance_openapi_catholic_polity implementation.

#![allow(unused_imports)]

use async_trait::async_trait;
use futures::{future, Stream, StreamExt, TryFutureExt, TryStreamExt};
use hyper::server::conn::Http;
use hyper::service::Service;
use log::info;
#[cfg(not(any(target_os = "macos", target_os = "windows", target_os = "ios")))]
use openssl::ssl::SslAcceptorBuilder;
use std::future::Future;
use std::marker::PhantomData;
use std::net::SocketAddr;
use std::sync::{Arc, Mutex};
use std::task::{Context, Poll};
use swagger::{Has, XSpanIdString};
use swagger::auth::MakeAllowAllAuthenticator;
use swagger::EmptyContext;
use tokio::net::TcpListener;

#[cfg(not(any(target_os = "macos", target_os = "windows", target_os = "ios")))]
use openssl::ssl::{SslAcceptor, SslFiletype, SslMethod};

use hvcg_governance_openapi_catholic_polity::models;

/// Builds an SSL implementation for Simple HTTPS from some hard-coded file names
pub async fn create(addr: &str, https: bool) {
    let addr = addr.parse().expect("Failed to parse bind address");

    let server = Server::new();

    let service = MakeService::new(server);

    let service = MakeAllowAllAuthenticator::new(service, "cosmo");

    let mut service =
        hvcg_governance_openapi_catholic_polity::server::context::MakeAddContext::<_, EmptyContext>::new(
            service
        );

    if https {
        #[cfg(any(target_os = "macos", target_os = "windows", target_os = "ios"))]
        {
            unimplemented!("SSL is not implemented for the examples on MacOS, Windows or iOS");
        }

        #[cfg(not(any(target_os = "macos", target_os = "windows", target_os = "ios")))]
        {
            let mut ssl = SslAcceptor::mozilla_intermediate_v5(SslMethod::tls()).expect("Failed to create SSL Acceptor");

            // Server authentication
            ssl.set_private_key_file("examples/server-key.pem", SslFiletype::PEM).expect("Failed to set private key");
            ssl.set_certificate_chain_file("examples/server-chain.pem").expect("Failed to set cerificate chain");
            ssl.check_private_key().expect("Failed to check private key");

            let tls_acceptor = Arc::new(ssl.build());
            let mut tcp_listener = TcpListener::bind(&addr).await.unwrap();
            let mut incoming = tcp_listener.incoming();

            while let (Some(tcp), rest) = incoming.into_future().await {
                if let Ok(tcp) = tcp {
                    let addr = tcp.peer_addr().expect("Unable to get remote address");
                    let service = service.call(addr);
                    let tls_acceptor = Arc::clone(&tls_acceptor);

                    tokio::spawn(async move {
                        let tls = tokio_openssl::accept(&*tls_acceptor, tcp).await.map_err(|_| ())?;

                        let service = service.await.map_err(|_| ())?;

                        Http::new().serve_connection(tls, service).await.map_err(|_| ())
                    });
                }

                incoming = rest;
            }
        }
    } else {
        // Using HTTP
        hyper::server::Server::bind(&addr).serve(service).await.unwrap()
    }
}

#[derive(Copy, Clone)]
pub struct Server<C> {
    marker: PhantomData<C>,
}

impl<C> Server<C> {
    pub fn new() -> Self {
        Server{marker: PhantomData}
    }
}


use hvcg_governance_openapi_catholic_polity::{
    Api,
    AddDeaneryResponse,
    DeleteDeaneryResponse,
    GetDeaneriesResponse,
    GetDeaneryByIdResponse,
    UpdateDeaneryResponse,
    AddDioceseResponse,
    DeleteDioceseResponse,
    GetDioceseByIdResponse,
    GetDiocesesResponse,
    UpdateDioceseResponse,
    AddInstituteResponse,
    DeleteInstituteResponse,
    GetInstituteByIdResponse,
    GetInstitutesResponse,
    UpdateInstituteResponse,
    AddParishResponse,
    DeleteParishResponse,
    GetParishByIdResponse,
    GetParishesResponse,
    UpdateParishResponse,
    AddProvinceResponse,
    DeleteProvinceResponse,
    GetProvinceByIdResponse,
    GetProvincesResponse,
    UpdateProvinceResponse,
};
use hvcg_governance_openapi_catholic_polity::server::MakeService;
use std::error::Error;
use swagger::ApiError;

#[async_trait]
impl<C> Api<C> for Server<C> where C: Has<XSpanIdString> + Send + Sync
{
    /// Add new deanery
    async fn add_deanery(
        &self,
        deanery: models::Deanery,
        context: &C) -> Result<AddDeaneryResponse, ApiError>
    {
        let context = context.clone();
        info!("add_deanery({:?}) - X-Span-ID: {:?}", deanery, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Delete deanery by ID
    async fn delete_deanery(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteDeaneryResponse, ApiError>
    {
        let context = context.clone();
        info!("delete_deanery({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Get deaneries
    async fn get_deaneries(
        &self,
        diocese_id: Option<uuid::Uuid>,
        name: Option<String>,
        offset: Option<i32>,
        count: Option<i32>,
        context: &C) -> Result<GetDeaneriesResponse, ApiError>
    {
        let context = context.clone();
        info!("get_deaneries({:?}, {:?}, {:?}, {:?}) - X-Span-ID: {:?}", diocese_id, name, offset, count, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Find deanery by ID
    async fn get_deanery_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetDeaneryByIdResponse, ApiError>
    {
        let context = context.clone();
        info!("get_deanery_by_id({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Update an existing deanery
    async fn update_deanery(
        &self,
        id: uuid::Uuid,
        deanery: models::Deanery,
        context: &C) -> Result<UpdateDeaneryResponse, ApiError>
    {
        let context = context.clone();
        info!("update_deanery({:?}, {:?}) - X-Span-ID: {:?}", id, deanery, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Add new diocese
    async fn add_diocese(
        &self,
        diocese: models::Diocese,
        context: &C) -> Result<AddDioceseResponse, ApiError>
    {
        let context = context.clone();
        info!("add_diocese({:?}) - X-Span-ID: {:?}", diocese, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Delete diocese by ID
    async fn delete_diocese(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteDioceseResponse, ApiError>
    {
        let context = context.clone();
        info!("delete_diocese({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Find diocese by ID
    async fn get_diocese_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetDioceseByIdResponse, ApiError>
    {
        let context = context.clone();
        info!("get_diocese_by_id({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Get all dioceses
    async fn get_dioceses(
        &self,
        province_id: Option<uuid::Uuid>,
        province_code: Option<models::Code>,
        name: Option<String>,
        location_name: Option<String>,
        location_address: Option<String>,
        location_email: Option<String>,
        sorts: Option<&Vec<models::DioceseSortCriteria>>,
        offset: Option<i32>,
        count: Option<i32>,
        context: &C) -> Result<GetDiocesesResponse, ApiError>
    {
        let context = context.clone();
        info!("get_dioceses({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}) - X-Span-ID: {:?}", province_id, province_code, name, location_name, location_address, location_email, sorts, offset, count, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Update an existing diocese
    async fn update_diocese(
        &self,
        id: uuid::Uuid,
        diocese: models::Diocese,
        context: &C) -> Result<UpdateDioceseResponse, ApiError>
    {
        let context = context.clone();
        info!("update_diocese({:?}, {:?}) - X-Span-ID: {:?}", id, diocese, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Add new institute
    async fn add_institute(
        &self,
        institute: models::Institute,
        context: &C) -> Result<AddInstituteResponse, ApiError>
    {
        let context = context.clone();
        info!("add_institute({:?}) - X-Span-ID: {:?}", institute, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Delete institute by ID
    async fn delete_institute(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteInstituteResponse, ApiError>
    {
        let context = context.clone();
        info!("delete_institute({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Find institute by ID
    async fn get_institute_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetInstituteByIdResponse, ApiError>
    {
        let context = context.clone();
        info!("get_institute_by_id({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Get institutes
    async fn get_institutes(
        &self,
        name: Option<String>,
        offset: Option<i32>,
        count: Option<i32>,
        context: &C) -> Result<GetInstitutesResponse, ApiError>
    {
        let context = context.clone();
        info!("get_institutes({:?}, {:?}, {:?}) - X-Span-ID: {:?}", name, offset, count, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Update an existing institute
    async fn update_institute(
        &self,
        id: uuid::Uuid,
        institute: models::Institute,
        context: &C) -> Result<UpdateInstituteResponse, ApiError>
    {
        let context = context.clone();
        info!("update_institute({:?}, {:?}) - X-Span-ID: {:?}", id, institute, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Add new parish
    async fn add_parish(
        &self,
        parish: Option<models::Parish>,
        context: &C) -> Result<AddParishResponse, ApiError>
    {
        let context = context.clone();
        info!("add_parish({:?}) - X-Span-ID: {:?}", parish, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Delete parish by ID
    async fn delete_parish(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteParishResponse, ApiError>
    {
        let context = context.clone();
        info!("delete_parish({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Find parish by ID
    async fn get_parish_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetParishByIdResponse, ApiError>
    {
        let context = context.clone();
        info!("get_parish_by_id({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Get all parishes
    async fn get_parishes(
        &self,
        deanery_id: Option<uuid::Uuid>,
        diocese_id: Option<uuid::Uuid>,
        name: Option<String>,
        offset: Option<i32>,
        count: Option<i32>,
        context: &C) -> Result<GetParishesResponse, ApiError>
    {
        let context = context.clone();
        info!("get_parishes({:?}, {:?}, {:?}, {:?}, {:?}) - X-Span-ID: {:?}", deanery_id, diocese_id, name, offset, count, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Update an existing parish
    async fn update_parish(
        &self,
        id: uuid::Uuid,
        parish: Option<models::Parish>,
        context: &C) -> Result<UpdateParishResponse, ApiError>
    {
        let context = context.clone();
        info!("update_parish({:?}, {:?}) - X-Span-ID: {:?}", id, parish, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Add new province
    async fn add_province(
        &self,
        province: Option<models::Province>,
        context: &C) -> Result<AddProvinceResponse, ApiError>
    {
        let context = context.clone();
        info!("add_province({:?}) - X-Span-ID: {:?}", province, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Delete province by ID
    async fn delete_province(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteProvinceResponse, ApiError>
    {
        let context = context.clone();
        info!("delete_province({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Find province by ID
    async fn get_province_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetProvinceByIdResponse, ApiError>
    {
        let context = context.clone();
        info!("get_province_by_id({:?}) - X-Span-ID: {:?}", id, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Get all provinces
    async fn get_provinces(
        &self,
        name: Option<String>,
        province_code: Option<models::Code>,
        offset: Option<i32>,
        count: Option<i32>,
        context: &C) -> Result<GetProvincesResponse, ApiError>
    {
        let context = context.clone();
        info!("get_provinces({:?}, {:?}, {:?}, {:?}) - X-Span-ID: {:?}", name, province_code, offset, count, context.get().0.clone());
        Err("Generic failuare".into())
    }

    /// Update an existing province
    async fn update_province(
        &self,
        id: uuid::Uuid,
        province: Option<models::Province>,
        context: &C) -> Result<UpdateProvinceResponse, ApiError>
    {
        let context = context.clone();
        info!("update_province({:?}, {:?}) - X-Span-ID: {:?}", id, province, context.get().0.clone());
        Err("Generic failuare".into())
    }

}
