#![allow(unused_qualifications)]

use crate::models;
#[cfg(any(feature = "client", feature = "server"))]
use crate::header;

/// Enumeration of values.
/// Since this enum's variants do not hold data, we can easily define them them as `#[repr(C)]`
/// which helps with FFI.
#[allow(non_camel_case_types)]
#[repr(C)]
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk_enum_derive::LabelledGenericEnum))]
pub enum Code {
    #[serde(rename = "HA_NOI")]
    HA_NOI,
    #[serde(rename = "HUE")]
    HUE,
    #[serde(rename = "SAI_GON")]
    SAI_GON,
}

impl std::fmt::Display for Code {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match *self {
            Code::HA_NOI => write!(f, "{}", "HA_NOI"),
            Code::HUE => write!(f, "{}", "HUE"),
            Code::SAI_GON => write!(f, "{}", "SAI_GON"),
        }
    }
}

impl std::str::FromStr for Code {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        match s {
            "HA_NOI" => std::result::Result::Ok(Code::HA_NOI),
            "HUE" => std::result::Result::Ok(Code::HUE),
            "SAI_GON" => std::result::Result::Ok(Code::SAI_GON),
            _ => std::result::Result::Err(format!("Value not valid: {}", s)),
        }
    }
}

/// An area controlled by a diocese
#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct Deanery {
    #[serde(rename = "diocese")]
    pub diocese: models::Diocese,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl Deanery {
    pub fn new(diocese: models::Diocese, name: String, ) -> Deanery {
        Deanery {
            diocese: diocese,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the Deanery value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for Deanery {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping diocese in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a Deanery value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for Deanery {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub diocese: Vec<models::Diocese>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing Deanery".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "diocese" => intermediate_rep.diocese.push(<models::Diocese as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing Deanery".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(Deanery {
            diocese: intermediate_rep.diocese.into_iter().next().ok_or("diocese missing in Deanery".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in Deanery".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<Deanery> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<Deanery>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<Deanery>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for Deanery - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<Deanery> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <Deanery as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into Deanery - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct DeaneryCollection {
    #[serde(rename = "deaneries")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub deaneries: Option<Vec<models::Deanery>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

}

impl DeaneryCollection {
    pub fn new() -> DeaneryCollection {
        DeaneryCollection {
            deaneries: None,
            has_more: None,
        }
    }
}

/// Converts the DeaneryCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for DeaneryCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping deaneries in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a DeaneryCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for DeaneryCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub deaneries: Vec<Vec<models::Deanery>>,
            pub has_more: Vec<bool>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing DeaneryCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "deaneries" => return std::result::Result::Err("Parsing a container in this style is not supported in DeaneryCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing DeaneryCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(DeaneryCollection {
            deaneries: intermediate_rep.deaneries.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<DeaneryCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<DeaneryCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<DeaneryCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for DeaneryCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<DeaneryCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <DeaneryCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into DeaneryCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


/// An area controlled by a ecclesiastical province
#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct Diocese {
    #[serde(rename = "province")]
    pub province: models::Province,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl Diocese {
    pub fn new(province: models::Province, name: String, ) -> Diocese {
        Diocese {
            province: province,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the Diocese value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for Diocese {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping province in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a Diocese value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for Diocese {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub province: Vec<models::Province>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing Diocese".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "province" => intermediate_rep.province.push(<models::Province as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing Diocese".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(Diocese {
            province: intermediate_rep.province.into_iter().next().ok_or("province missing in Diocese".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in Diocese".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<Diocese> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<Diocese>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<Diocese>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for Diocese - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<Diocese> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <Diocese as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into Diocese - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct DioceseCollection {
    #[serde(rename = "dioceses")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub dioceses: Option<Vec<models::Diocese>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

}

impl DioceseCollection {
    pub fn new() -> DioceseCollection {
        DioceseCollection {
            dioceses: None,
            has_more: None,
        }
    }
}

/// Converts the DioceseCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for DioceseCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping dioceses in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a DioceseCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for DioceseCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub dioceses: Vec<Vec<models::Diocese>>,
            pub has_more: Vec<bool>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing DioceseCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "dioceses" => return std::result::Result::Err("Parsing a container in this style is not supported in DioceseCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing DioceseCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(DioceseCollection {
            dioceses: intermediate_rep.dioceses.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<DioceseCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<DioceseCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<DioceseCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for DioceseCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<DioceseCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <DioceseCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into DioceseCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct Institute {
    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl Institute {
    pub fn new(name: String, ) -> Institute {
        Institute {
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the Institute value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for Institute {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a Institute value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for Institute {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing Institute".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing Institute".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(Institute {
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in Institute".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<Institute> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<Institute>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<Institute>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for Institute - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<Institute> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <Institute as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into Institute - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct InstituteCollection {
    #[serde(rename = "institutes")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub institutes: Option<Vec<models::Institute>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

}

impl InstituteCollection {
    pub fn new() -> InstituteCollection {
        InstituteCollection {
            institutes: None,
            has_more: None,
        }
    }
}

/// Converts the InstituteCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for InstituteCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping institutes in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a InstituteCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for InstituteCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub institutes: Vec<Vec<models::Institute>>,
            pub has_more: Vec<bool>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing InstituteCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "institutes" => return std::result::Result::Err("Parsing a container in this style is not supported in InstituteCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing InstituteCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(InstituteCollection {
            institutes: intermediate_rep.institutes.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<InstituteCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<InstituteCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<InstituteCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for InstituteCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<InstituteCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <InstituteCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into InstituteCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct Parish {
    #[serde(rename = "deanery")]
    pub deanery: models::Deanery,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl Parish {
    pub fn new(deanery: models::Deanery, name: String, ) -> Parish {
        Parish {
            deanery: deanery,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the Parish value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for Parish {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping deanery in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a Parish value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for Parish {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub deanery: Vec<models::Deanery>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing Parish".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "deanery" => intermediate_rep.deanery.push(<models::Deanery as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing Parish".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(Parish {
            deanery: intermediate_rep.deanery.into_iter().next().ok_or("deanery missing in Parish".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in Parish".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<Parish> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<Parish>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<Parish>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for Parish - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<Parish> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <Parish as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into Parish - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct ParishCollection {
    #[serde(rename = "parishes")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub parishes: Option<Vec<models::Parish>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

}

impl ParishCollection {
    pub fn new() -> ParishCollection {
        ParishCollection {
            parishes: None,
            has_more: None,
        }
    }
}

/// Converts the ParishCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for ParishCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping parishes in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a ParishCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for ParishCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub parishes: Vec<Vec<models::Parish>>,
            pub has_more: Vec<bool>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing ParishCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "parishes" => return std::result::Result::Err("Parsing a container in this style is not supported in ParishCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing ParishCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(ParishCollection {
            parishes: intermediate_rep.parishes.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<ParishCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<ParishCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<ParishCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for ParishCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<ParishCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <ParishCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into ParishCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct Polity {
    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl Polity {
    pub fn new(name: String, ) -> Polity {
        Polity {
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the Polity value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for Polity {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a Polity value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for Polity {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing Polity".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing Polity".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(Polity {
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in Polity".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<Polity> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<Polity>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<Polity>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for Polity - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<Polity> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <Polity as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into Polity - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct Province {
    #[serde(rename = "code")]
    pub code: models::Code,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl Province {
    pub fn new(code: models::Code, name: String, ) -> Province {
        Province {
            code: code,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the Province value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for Province {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping code in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a Province value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for Province {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub code: Vec<models::Code>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing Province".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "code" => intermediate_rep.code.push(<models::Code as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing Province".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(Province {
            code: intermediate_rep.code.into_iter().next().ok_or("code missing in Province".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in Province".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<Province> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<Province>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<Province>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for Province - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<Province> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <Province as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into Province - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct ProvinceCollection {
    #[serde(rename = "provinces")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub provinces: Option<Vec<models::Province>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

}

impl ProvinceCollection {
    pub fn new() -> ProvinceCollection {
        ProvinceCollection {
            provinces: None,
            has_more: None,
        }
    }
}

/// Converts the ProvinceCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for ProvinceCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping provinces in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a ProvinceCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for ProvinceCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub provinces: Vec<Vec<models::Province>>,
            pub has_more: Vec<bool>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing ProvinceCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "provinces" => return std::result::Result::Err("Parsing a container in this style is not supported in ProvinceCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing ProvinceCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(ProvinceCollection {
            provinces: intermediate_rep.provinces.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<ProvinceCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<ProvinceCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<ProvinceCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for ProvinceCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<ProvinceCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <ProvinceCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into ProvinceCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}

