#![allow(unused_qualifications)]

use crate::models;
#[cfg(any(feature = "client", feature = "server"))]
use crate::header;

/// Enumeration of values.
/// Since this enum's variants do not hold data, we can easily define them them as `#[repr(C)]`
/// which helps with FFI.
#[allow(non_camel_case_types)]
#[repr(C)]
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk_enum_derive::LabelledGenericEnum))]
pub enum Code {
    #[serde(rename = "HA_NOI")]
    HA_NOI,
    #[serde(rename = "HUE")]
    HUE,
    #[serde(rename = "SAI_GON")]
    SAI_GON,
}

impl std::fmt::Display for Code {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match *self {
            Code::HA_NOI => write!(f, "{}", "HA_NOI"),
            Code::HUE => write!(f, "{}", "HUE"),
            Code::SAI_GON => write!(f, "{}", "SAI_GON"),
        }
    }
}

impl std::str::FromStr for Code {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        match s {
            "HA_NOI" => std::result::Result::Ok(Code::HA_NOI),
            "HUE" => std::result::Result::Ok(Code::HUE),
            "SAI_GON" => std::result::Result::Ok(Code::SAI_GON),
            _ => std::result::Result::Err(format!("Value not valid: {}", s)),
        }
    }
}

#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct DeaneryCollection {
    #[serde(rename = "deaneries")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub deaneries: Option<Vec<models::DeaneryView>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

    #[serde(rename = "total")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub total: Option<i64>,

}

impl DeaneryCollection {
    pub fn new() -> DeaneryCollection {
        DeaneryCollection {
            deaneries: None,
            has_more: None,
            total: None,
        }
    }
}

/// Converts the DeaneryCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for DeaneryCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping deaneries in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }


        if let Some(ref total) = self.total {
            params.push("total".to_string());
            params.push(total.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a DeaneryCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for DeaneryCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub deaneries: Vec<Vec<models::DeaneryView>>,
            pub has_more: Vec<bool>,
            pub total: Vec<i64>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing DeaneryCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "deaneries" => return std::result::Result::Err("Parsing a container in this style is not supported in DeaneryCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "total" => intermediate_rep.total.push(<i64 as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing DeaneryCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(DeaneryCollection {
            deaneries: intermediate_rep.deaneries.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
            total: intermediate_rep.total.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<DeaneryCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<DeaneryCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<DeaneryCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for DeaneryCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<DeaneryCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <DeaneryCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into DeaneryCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


/// An area controlled by a diocese
#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct DeaneryMutation {
    #[serde(rename = "dioceseId")]
    pub diocese_id: uuid::Uuid,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl DeaneryMutation {
    pub fn new(diocese_id: uuid::Uuid, name: String, ) -> DeaneryMutation {
        DeaneryMutation {
            diocese_id: diocese_id,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the DeaneryMutation value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for DeaneryMutation {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping dioceseId in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a DeaneryMutation value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for DeaneryMutation {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub diocese_id: Vec<uuid::Uuid>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing DeaneryMutation".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "dioceseId" => intermediate_rep.diocese_id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing DeaneryMutation".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(DeaneryMutation {
            diocese_id: intermediate_rep.diocese_id.into_iter().next().ok_or("dioceseId missing in DeaneryMutation".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in DeaneryMutation".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<DeaneryMutation> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<DeaneryMutation>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<DeaneryMutation>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for DeaneryMutation - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<DeaneryMutation> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <DeaneryMutation as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into DeaneryMutation - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


/// An area controlled by a diocese
#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct DeaneryView {
    #[serde(rename = "diocese")]
    pub diocese: models::DioceseView,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl DeaneryView {
    pub fn new(diocese: models::DioceseView, name: String, ) -> DeaneryView {
        DeaneryView {
            diocese: diocese,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the DeaneryView value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for DeaneryView {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping diocese in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a DeaneryView value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for DeaneryView {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub diocese: Vec<models::DioceseView>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing DeaneryView".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "diocese" => intermediate_rep.diocese.push(<models::DioceseView as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing DeaneryView".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(DeaneryView {
            diocese: intermediate_rep.diocese.into_iter().next().ok_or("diocese missing in DeaneryView".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in DeaneryView".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<DeaneryView> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<DeaneryView>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<DeaneryView>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for DeaneryView - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<DeaneryView> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <DeaneryView as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into DeaneryView - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct DioceseCollection {
    #[serde(rename = "dioceses")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub dioceses: Option<Vec<models::DioceseView>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

    #[serde(rename = "total")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub total: Option<i64>,

}

impl DioceseCollection {
    pub fn new() -> DioceseCollection {
        DioceseCollection {
            dioceses: None,
            has_more: None,
            total: None,
        }
    }
}

/// Converts the DioceseCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for DioceseCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping dioceses in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }


        if let Some(ref total) = self.total {
            params.push("total".to_string());
            params.push(total.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a DioceseCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for DioceseCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub dioceses: Vec<Vec<models::DioceseView>>,
            pub has_more: Vec<bool>,
            pub total: Vec<i64>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing DioceseCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "dioceses" => return std::result::Result::Err("Parsing a container in this style is not supported in DioceseCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "total" => intermediate_rep.total.push(<i64 as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing DioceseCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(DioceseCollection {
            dioceses: intermediate_rep.dioceses.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
            total: intermediate_rep.total.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<DioceseCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<DioceseCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<DioceseCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for DioceseCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<DioceseCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <DioceseCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into DioceseCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


/// An area controlled by a ecclesiastical province
#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct DioceseMutation {
    #[serde(rename = "provinceId")]
    pub province_id: uuid::Uuid,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl DioceseMutation {
    pub fn new(province_id: uuid::Uuid, name: String, ) -> DioceseMutation {
        DioceseMutation {
            province_id: province_id,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the DioceseMutation value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for DioceseMutation {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping provinceId in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a DioceseMutation value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for DioceseMutation {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub province_id: Vec<uuid::Uuid>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing DioceseMutation".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "provinceId" => intermediate_rep.province_id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing DioceseMutation".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(DioceseMutation {
            province_id: intermediate_rep.province_id.into_iter().next().ok_or("provinceId missing in DioceseMutation".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in DioceseMutation".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<DioceseMutation> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<DioceseMutation>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<DioceseMutation>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for DioceseMutation - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<DioceseMutation> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <DioceseMutation as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into DioceseMutation - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


/// Enumeration of values.
/// Since this enum's variants do not hold data, we can easily define them them as `#[repr(C)]`
/// which helps with FFI.
#[allow(non_camel_case_types)]
#[repr(C)]
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk_enum_derive::LabelledGenericEnum))]
pub enum DioceseSortCriteria {
    #[serde(rename = "NAME_ASC")]
    NAME_ASC,
    #[serde(rename = "NAME_DESC")]
    NAME_DESC,
    #[serde(rename = "LOCATION_NAME_ASC")]
    LOCATION_NAME_ASC,
    #[serde(rename = "LOCATION_NAME_DESC")]
    LOCATION_NAME_DESC,
    #[serde(rename = "LOCATION_EMAIL_ASC")]
    LOCATION_EMAIL_ASC,
    #[serde(rename = "LOCATION_EMAIL_DESC")]
    LOCATION_EMAIL_DESC,
    #[serde(rename = "LOCATION_ADDRESS_ASC")]
    LOCATION_ADDRESS_ASC,
    #[serde(rename = "LOCATION_ADDRESS_DESC")]
    LOCATION_ADDRESS_DESC,
}

impl std::fmt::Display for DioceseSortCriteria {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match *self {
            DioceseSortCriteria::NAME_ASC => write!(f, "{}", "NAME_ASC"),
            DioceseSortCriteria::NAME_DESC => write!(f, "{}", "NAME_DESC"),
            DioceseSortCriteria::LOCATION_NAME_ASC => write!(f, "{}", "LOCATION_NAME_ASC"),
            DioceseSortCriteria::LOCATION_NAME_DESC => write!(f, "{}", "LOCATION_NAME_DESC"),
            DioceseSortCriteria::LOCATION_EMAIL_ASC => write!(f, "{}", "LOCATION_EMAIL_ASC"),
            DioceseSortCriteria::LOCATION_EMAIL_DESC => write!(f, "{}", "LOCATION_EMAIL_DESC"),
            DioceseSortCriteria::LOCATION_ADDRESS_ASC => write!(f, "{}", "LOCATION_ADDRESS_ASC"),
            DioceseSortCriteria::LOCATION_ADDRESS_DESC => write!(f, "{}", "LOCATION_ADDRESS_DESC"),
        }
    }
}

impl std::str::FromStr for DioceseSortCriteria {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        match s {
            "NAME_ASC" => std::result::Result::Ok(DioceseSortCriteria::NAME_ASC),
            "NAME_DESC" => std::result::Result::Ok(DioceseSortCriteria::NAME_DESC),
            "LOCATION_NAME_ASC" => std::result::Result::Ok(DioceseSortCriteria::LOCATION_NAME_ASC),
            "LOCATION_NAME_DESC" => std::result::Result::Ok(DioceseSortCriteria::LOCATION_NAME_DESC),
            "LOCATION_EMAIL_ASC" => std::result::Result::Ok(DioceseSortCriteria::LOCATION_EMAIL_ASC),
            "LOCATION_EMAIL_DESC" => std::result::Result::Ok(DioceseSortCriteria::LOCATION_EMAIL_DESC),
            "LOCATION_ADDRESS_ASC" => std::result::Result::Ok(DioceseSortCriteria::LOCATION_ADDRESS_ASC),
            "LOCATION_ADDRESS_DESC" => std::result::Result::Ok(DioceseSortCriteria::LOCATION_ADDRESS_DESC),
            _ => std::result::Result::Err(format!("Value not valid: {}", s)),
        }
    }
}

/// An area controlled by a ecclesiastical province
#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct DioceseView {
    #[serde(rename = "province")]
    pub province: models::ProvinceView,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl DioceseView {
    pub fn new(province: models::ProvinceView, name: String, ) -> DioceseView {
        DioceseView {
            province: province,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the DioceseView value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for DioceseView {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping province in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a DioceseView value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for DioceseView {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub province: Vec<models::ProvinceView>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing DioceseView".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "province" => intermediate_rep.province.push(<models::ProvinceView as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing DioceseView".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(DioceseView {
            province: intermediate_rep.province.into_iter().next().ok_or("province missing in DioceseView".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in DioceseView".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<DioceseView> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<DioceseView>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<DioceseView>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for DioceseView - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<DioceseView> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <DioceseView as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into DioceseView - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct InstituteCollection {
    #[serde(rename = "institutes")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub institutes: Option<Vec<models::InstituteView>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

    #[serde(rename = "total")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub total: Option<i64>,

}

impl InstituteCollection {
    pub fn new() -> InstituteCollection {
        InstituteCollection {
            institutes: None,
            has_more: None,
            total: None,
        }
    }
}

/// Converts the InstituteCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for InstituteCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping institutes in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }


        if let Some(ref total) = self.total {
            params.push("total".to_string());
            params.push(total.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a InstituteCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for InstituteCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub institutes: Vec<Vec<models::InstituteView>>,
            pub has_more: Vec<bool>,
            pub total: Vec<i64>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing InstituteCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "institutes" => return std::result::Result::Err("Parsing a container in this style is not supported in InstituteCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "total" => intermediate_rep.total.push(<i64 as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing InstituteCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(InstituteCollection {
            institutes: intermediate_rep.institutes.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
            total: intermediate_rep.total.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<InstituteCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<InstituteCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<InstituteCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for InstituteCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<InstituteCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <InstituteCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into InstituteCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct InstituteMutation {
    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl InstituteMutation {
    pub fn new(name: String, ) -> InstituteMutation {
        InstituteMutation {
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the InstituteMutation value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for InstituteMutation {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a InstituteMutation value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for InstituteMutation {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing InstituteMutation".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing InstituteMutation".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(InstituteMutation {
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in InstituteMutation".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<InstituteMutation> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<InstituteMutation>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<InstituteMutation>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for InstituteMutation - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<InstituteMutation> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <InstituteMutation as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into InstituteMutation - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


/// Enumeration of values.
/// Since this enum's variants do not hold data, we can easily define them them as `#[repr(C)]`
/// which helps with FFI.
#[allow(non_camel_case_types)]
#[repr(C)]
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk_enum_derive::LabelledGenericEnum))]
pub enum InstituteSortCriteria {
    #[serde(rename = "NAME_ASC")]
    NAME_ASC,
    #[serde(rename = "NAME_DESC")]
    NAME_DESC,
    #[serde(rename = "LOCATION_NAME_ASC")]
    LOCATION_NAME_ASC,
    #[serde(rename = "LOCATION_NAME_DESC")]
    LOCATION_NAME_DESC,
    #[serde(rename = "LOCATION_EMAIL_ASC")]
    LOCATION_EMAIL_ASC,
    #[serde(rename = "LOCATION_EMAIL_DESC")]
    LOCATION_EMAIL_DESC,
    #[serde(rename = "LOCATION_ADDRESS_ASC")]
    LOCATION_ADDRESS_ASC,
    #[serde(rename = "LOCATION_ADDRESS_DESC")]
    LOCATION_ADDRESS_DESC,
}

impl std::fmt::Display for InstituteSortCriteria {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match *self {
            InstituteSortCriteria::NAME_ASC => write!(f, "{}", "NAME_ASC"),
            InstituteSortCriteria::NAME_DESC => write!(f, "{}", "NAME_DESC"),
            InstituteSortCriteria::LOCATION_NAME_ASC => write!(f, "{}", "LOCATION_NAME_ASC"),
            InstituteSortCriteria::LOCATION_NAME_DESC => write!(f, "{}", "LOCATION_NAME_DESC"),
            InstituteSortCriteria::LOCATION_EMAIL_ASC => write!(f, "{}", "LOCATION_EMAIL_ASC"),
            InstituteSortCriteria::LOCATION_EMAIL_DESC => write!(f, "{}", "LOCATION_EMAIL_DESC"),
            InstituteSortCriteria::LOCATION_ADDRESS_ASC => write!(f, "{}", "LOCATION_ADDRESS_ASC"),
            InstituteSortCriteria::LOCATION_ADDRESS_DESC => write!(f, "{}", "LOCATION_ADDRESS_DESC"),
        }
    }
}

impl std::str::FromStr for InstituteSortCriteria {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        match s {
            "NAME_ASC" => std::result::Result::Ok(InstituteSortCriteria::NAME_ASC),
            "NAME_DESC" => std::result::Result::Ok(InstituteSortCriteria::NAME_DESC),
            "LOCATION_NAME_ASC" => std::result::Result::Ok(InstituteSortCriteria::LOCATION_NAME_ASC),
            "LOCATION_NAME_DESC" => std::result::Result::Ok(InstituteSortCriteria::LOCATION_NAME_DESC),
            "LOCATION_EMAIL_ASC" => std::result::Result::Ok(InstituteSortCriteria::LOCATION_EMAIL_ASC),
            "LOCATION_EMAIL_DESC" => std::result::Result::Ok(InstituteSortCriteria::LOCATION_EMAIL_DESC),
            "LOCATION_ADDRESS_ASC" => std::result::Result::Ok(InstituteSortCriteria::LOCATION_ADDRESS_ASC),
            "LOCATION_ADDRESS_DESC" => std::result::Result::Ok(InstituteSortCriteria::LOCATION_ADDRESS_DESC),
            _ => std::result::Result::Err(format!("Value not valid: {}", s)),
        }
    }
}

#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct InstituteView {
    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl InstituteView {
    pub fn new(name: String, ) -> InstituteView {
        InstituteView {
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the InstituteView value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for InstituteView {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a InstituteView value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for InstituteView {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing InstituteView".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing InstituteView".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(InstituteView {
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in InstituteView".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<InstituteView> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<InstituteView>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<InstituteView>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for InstituteView - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<InstituteView> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <InstituteView as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into InstituteView - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct ParishCollection {
    #[serde(rename = "parishes")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub parishes: Option<Vec<models::ParishView>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

    #[serde(rename = "total")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub total: Option<i64>,

}

impl ParishCollection {
    pub fn new() -> ParishCollection {
        ParishCollection {
            parishes: None,
            has_more: None,
            total: None,
        }
    }
}

/// Converts the ParishCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for ParishCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping parishes in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }


        if let Some(ref total) = self.total {
            params.push("total".to_string());
            params.push(total.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a ParishCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for ParishCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub parishes: Vec<Vec<models::ParishView>>,
            pub has_more: Vec<bool>,
            pub total: Vec<i64>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing ParishCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "parishes" => return std::result::Result::Err("Parsing a container in this style is not supported in ParishCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "total" => intermediate_rep.total.push(<i64 as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing ParishCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(ParishCollection {
            parishes: intermediate_rep.parishes.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
            total: intermediate_rep.total.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<ParishCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<ParishCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<ParishCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for ParishCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<ParishCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <ParishCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into ParishCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct ParishMutation {
    #[serde(rename = "deaneryId")]
    pub deanery_id: uuid::Uuid,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl ParishMutation {
    pub fn new(deanery_id: uuid::Uuid, name: String, ) -> ParishMutation {
        ParishMutation {
            deanery_id: deanery_id,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the ParishMutation value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for ParishMutation {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping deaneryId in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a ParishMutation value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for ParishMutation {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub deanery_id: Vec<uuid::Uuid>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing ParishMutation".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "deaneryId" => intermediate_rep.deanery_id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing ParishMutation".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(ParishMutation {
            deanery_id: intermediate_rep.deanery_id.into_iter().next().ok_or("deaneryId missing in ParishMutation".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in ParishMutation".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<ParishMutation> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<ParishMutation>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<ParishMutation>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for ParishMutation - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<ParishMutation> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <ParishMutation as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into ParishMutation - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct ParishView {
    #[serde(rename = "deanery")]
    pub deanery: models::DeaneryView,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl ParishView {
    pub fn new(deanery: models::DeaneryView, name: String, ) -> ParishView {
        ParishView {
            deanery: deanery,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the ParishView value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for ParishView {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping deanery in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a ParishView value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for ParishView {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub deanery: Vec<models::DeaneryView>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing ParishView".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "deanery" => intermediate_rep.deanery.push(<models::DeaneryView as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing ParishView".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(ParishView {
            deanery: intermediate_rep.deanery.into_iter().next().ok_or("deanery missing in ParishView".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in ParishView".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<ParishView> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<ParishView>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<ParishView>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for ParishView - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<ParishView> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <ParishView as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into ParishView - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct Polity {
    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl Polity {
    pub fn new(name: String, ) -> Polity {
        Polity {
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the Polity value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for Polity {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a Polity value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for Polity {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing Polity".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing Polity".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(Polity {
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in Polity".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<Polity> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<Polity>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<Polity>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for Polity - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<Polity> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <Polity as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into Polity - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct ProvinceCollection {
    #[serde(rename = "provinces")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub provinces: Option<Vec<models::ProvinceView>>,

    #[serde(rename = "hasMore")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub has_more: Option<bool>,

    #[serde(rename = "total")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub total: Option<i64>,

}

impl ProvinceCollection {
    pub fn new() -> ProvinceCollection {
        ProvinceCollection {
            provinces: None,
            has_more: None,
            total: None,
        }
    }
}

/// Converts the ProvinceCollection value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for ProvinceCollection {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping provinces in query parameter serialization


        if let Some(ref has_more) = self.has_more {
            params.push("hasMore".to_string());
            params.push(has_more.to_string());
        }


        if let Some(ref total) = self.total {
            params.push("total".to_string());
            params.push(total.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a ProvinceCollection value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for ProvinceCollection {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub provinces: Vec<Vec<models::ProvinceView>>,
            pub has_more: Vec<bool>,
            pub total: Vec<i64>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing ProvinceCollection".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "provinces" => return std::result::Result::Err("Parsing a container in this style is not supported in ProvinceCollection".to_string()),
                    "hasMore" => intermediate_rep.has_more.push(<bool as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "total" => intermediate_rep.total.push(<i64 as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing ProvinceCollection".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(ProvinceCollection {
            provinces: intermediate_rep.provinces.into_iter().next(),
            has_more: intermediate_rep.has_more.into_iter().next(),
            total: intermediate_rep.total.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<ProvinceCollection> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<ProvinceCollection>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<ProvinceCollection>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for ProvinceCollection - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<ProvinceCollection> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <ProvinceCollection as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into ProvinceCollection - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct ProvinceMutation {
    #[serde(rename = "code")]
    pub code: models::Code,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl ProvinceMutation {
    pub fn new(code: models::Code, name: String, ) -> ProvinceMutation {
        ProvinceMutation {
            code: code,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the ProvinceMutation value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for ProvinceMutation {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping code in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a ProvinceMutation value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for ProvinceMutation {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub code: Vec<models::Code>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing ProvinceMutation".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "code" => intermediate_rep.code.push(<models::Code as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing ProvinceMutation".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(ProvinceMutation {
            code: intermediate_rep.code.into_iter().next().ok_or("code missing in ProvinceMutation".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in ProvinceMutation".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<ProvinceMutation> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<ProvinceMutation>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<ProvinceMutation>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for ProvinceMutation - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<ProvinceMutation> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <ProvinceMutation as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into ProvinceMutation - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}


#[derive(Debug, Clone, PartialEq, serde::Serialize, serde::Deserialize)]
#[cfg_attr(feature = "conversion", derive(frunk::LabelledGeneric))]
pub struct ProvinceView {
    #[serde(rename = "code")]
    pub code: models::Code,

    #[serde(rename = "id")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub id: Option<uuid::Uuid>,

    #[serde(rename = "name")]
    pub name: String,

    #[serde(rename = "locationEmail")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_email: Option<String>,

    #[serde(rename = "locationAddress")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_address: Option<String>,

    #[serde(rename = "locationName")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub location_name: Option<String>,

    #[serde(rename = "personInCharge")]
    #[serde(skip_serializing_if="Option::is_none")]
    pub person_in_charge: Option<String>,

}

impl ProvinceView {
    pub fn new(code: models::Code, name: String, ) -> ProvinceView {
        ProvinceView {
            code: code,
            id: None,
            name: name,
            location_email: None,
            location_address: None,
            location_name: None,
            person_in_charge: None,
        }
    }
}

/// Converts the ProvinceView value to the Query Parameters representation (style=form, explode=false)
/// specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde serializer
impl std::string::ToString for ProvinceView {
    fn to_string(&self) -> String {
        let mut params: Vec<String> = vec![];
        // Skipping code in query parameter serialization

        // Skipping id in query parameter serialization


        params.push("name".to_string());
        params.push(self.name.to_string());


        if let Some(ref location_email) = self.location_email {
            params.push("locationEmail".to_string());
            params.push(location_email.to_string());
        }


        if let Some(ref location_address) = self.location_address {
            params.push("locationAddress".to_string());
            params.push(location_address.to_string());
        }


        if let Some(ref location_name) = self.location_name {
            params.push("locationName".to_string());
            params.push(location_name.to_string());
        }


        if let Some(ref person_in_charge) = self.person_in_charge {
            params.push("personInCharge".to_string());
            params.push(person_in_charge.to_string());
        }

        params.join(",").to_string()
    }
}

/// Converts Query Parameters representation (style=form, explode=false) to a ProvinceView value
/// as specified in https://swagger.io/docs/specification/serialization/
/// Should be implemented in a serde deserializer
impl std::str::FromStr for ProvinceView {
    type Err = String;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        #[derive(Default)]
        // An intermediate representation of the struct to use for parsing.
        struct IntermediateRep {
            pub code: Vec<models::Code>,
            pub id: Vec<uuid::Uuid>,
            pub name: Vec<String>,
            pub location_email: Vec<String>,
            pub location_address: Vec<String>,
            pub location_name: Vec<String>,
            pub person_in_charge: Vec<String>,
        }

        let mut intermediate_rep = IntermediateRep::default();

        // Parse into intermediate representation
        let mut string_iter = s.split(',').into_iter();
        let mut key_result = string_iter.next();

        while key_result.is_some() {
            let val = match string_iter.next() {
                Some(x) => x,
                None => return std::result::Result::Err("Missing value while parsing ProvinceView".to_string())
            };

            if let Some(key) = key_result {
                match key {
                    "code" => intermediate_rep.code.push(<models::Code as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "id" => intermediate_rep.id.push(<uuid::Uuid as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "name" => intermediate_rep.name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationEmail" => intermediate_rep.location_email.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationAddress" => intermediate_rep.location_address.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "locationName" => intermediate_rep.location_name.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    "personInCharge" => intermediate_rep.person_in_charge.push(<String as std::str::FromStr>::from_str(val).map_err(|x| format!("{}", x))?),
                    _ => return std::result::Result::Err("Unexpected key while parsing ProvinceView".to_string())
                }
            }

            // Get the next key
            key_result = string_iter.next();
        }

        // Use the intermediate representation to return the struct
        std::result::Result::Ok(ProvinceView {
            code: intermediate_rep.code.into_iter().next().ok_or("code missing in ProvinceView".to_string())?,
            id: intermediate_rep.id.into_iter().next(),
            name: intermediate_rep.name.into_iter().next().ok_or("name missing in ProvinceView".to_string())?,
            location_email: intermediate_rep.location_email.into_iter().next(),
            location_address: intermediate_rep.location_address.into_iter().next(),
            location_name: intermediate_rep.location_name.into_iter().next(),
            person_in_charge: intermediate_rep.person_in_charge.into_iter().next(),
        })
    }
}

// Methods for converting between header::IntoHeaderValue<ProvinceView> and hyper::header::HeaderValue

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<header::IntoHeaderValue<ProvinceView>> for hyper::header::HeaderValue {
    type Error = String;

    fn try_from(hdr_value: header::IntoHeaderValue<ProvinceView>) -> std::result::Result<Self, Self::Error> {
        let hdr_value = hdr_value.to_string();
        match hyper::header::HeaderValue::from_str(&hdr_value) {
             std::result::Result::Ok(value) => std::result::Result::Ok(value),
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Invalid header value for ProvinceView - value: {} is invalid {}",
                     hdr_value, e))
        }
    }
}

#[cfg(any(feature = "client", feature = "server"))]
impl std::convert::TryFrom<hyper::header::HeaderValue> for header::IntoHeaderValue<ProvinceView> {
    type Error = String;

    fn try_from(hdr_value: hyper::header::HeaderValue) -> std::result::Result<Self, Self::Error> {
        match hdr_value.to_str() {
             std::result::Result::Ok(value) => {
                    match <ProvinceView as std::str::FromStr>::from_str(value) {
                        std::result::Result::Ok(value) => std::result::Result::Ok(header::IntoHeaderValue(value)),
                        std::result::Result::Err(err) => std::result::Result::Err(
                            format!("Unable to convert header value '{}' into ProvinceView - {}",
                                value, err))
                    }
             },
             std::result::Result::Err(e) => std::result::Result::Err(
                 format!("Unable to convert header: {:?} to string: {}",
                     hdr_value, e))
        }
    }
}

