#![allow(missing_docs, trivial_casts, unused_variables, unused_mut, unused_imports, unused_extern_crates, non_camel_case_types)]

use async_trait::async_trait;
use futures::Stream;
use std::error::Error;
use std::task::{Poll, Context};
use swagger::{ApiError, ContextWrapper};
use serde::{Serialize, Deserialize};

type ServiceError = Box<dyn Error + Send + Sync + 'static>;

pub const BASE_PATH: &'static str = "";
pub const API_VERSION: &'static str = "0.1.1";

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum AddDeaneryResponse {
    /// successful operation
    SuccessfulOperation
    (models::Deanery)
    ,
    /// Invalid input
    InvalidInput
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
pub enum DeleteDeaneryResponse {
    /// Invalid deanery value
    InvalidDeaneryValue
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum GetAllDeaneriesResponse {
    /// successful operation
    SuccessfulOperation
    (models::Deanery)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Deanery not found
    DeaneryNotFound
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum GetDeaneryByIdResponse {
    /// successful operation
    SuccessfulOperation
    (models::Deanery)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Deanery not found
    DeaneryNotFound
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum UpdateDeaneryResponse {
    /// successful operation
    SuccessfulOperation
    (models::Parish)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Deanery not found
    DeaneryNotFound
    ,
    /// Validation exception
    ValidationException
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum AddDioceseResponse {
    /// successful operation
    SuccessfulOperation
    (models::Diocese)
    ,
    /// Invalid input
    InvalidInput
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
pub enum DeleteDioceseResponse {
    /// Invalid diocese value
    InvalidDioceseValue
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum GetAllDiocesesResponse {
    /// successful operation
    SuccessfulOperation
    (models::Diocese)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Diocese not found
    DioceseNotFound
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum GetDioceseByIdResponse {
    /// successful operation
    SuccessfulOperation
    (models::Diocese)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Diocese not found
    DioceseNotFound
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum UpdateDioceseResponse {
    /// successful operation
    SuccessfulOperation
    (models::Parish)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Diocese not found
    DioceseNotFound
    ,
    /// Validation exception
    ValidationException
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum AddParishResponse {
    /// successful operation
    SuccessfulOperation
    (models::Parish)
    ,
    /// Invalid input
    InvalidInput
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
pub enum DeleteParishResponse {
    /// Invalid parish value
    InvalidParishValue
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum GetAllParishesResponse {
    /// successful operation
    SuccessfulOperation
    (models::Parish)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Parish not found
    ParishNotFound
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum GetParishByIdResponse {
    /// successful operation
    SuccessfulOperation
    (models::Parish)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Parish not found
    ParishNotFound
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum UpdateParishResponse {
    /// successful operation
    SuccessfulOperation
    (models::Parish)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// Parish not found
    ParishNotFound
    ,
    /// Validation exception
    ValidationException
}

/// API
#[async_trait]
pub trait Api<C: Send + Sync> {
    fn poll_ready(&self, _cx: &mut Context) -> Poll<Result<(), Box<dyn Error + Send + Sync + 'static>>> {
        Poll::Ready(Ok(()))
    }

    /// Add new deanery
    async fn add_deanery(
        &self,
        deanery: models::Deanery,
        context: &C) -> Result<AddDeaneryResponse, ApiError>;

    /// Delete deanery by ID
    async fn delete_deanery(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteDeaneryResponse, ApiError>;

    /// Get all deaneries
    async fn get_all_deaneries(
        &self,
        province: Option<models::EcclesiasticalProvince>,
        name: Option<String>,
        context: &C) -> Result<GetAllDeaneriesResponse, ApiError>;

    /// Find deanery by ID
    async fn get_deanery_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetDeaneryByIdResponse, ApiError>;

    /// Update an existing deanery
    async fn update_deanery(
        &self,
        id: uuid::Uuid,
        deanery: models::Deanery,
        context: &C) -> Result<UpdateDeaneryResponse, ApiError>;

    /// Add new diocese
    async fn add_diocese(
        &self,
        diocese: models::Diocese,
        context: &C) -> Result<AddDioceseResponse, ApiError>;

    /// Delete diocese by ID
    async fn delete_diocese(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteDioceseResponse, ApiError>;

    /// Get all dioceses
    async fn get_all_dioceses(
        &self,
        province: Option<models::EcclesiasticalProvince>,
        name: Option<String>,
        context: &C) -> Result<GetAllDiocesesResponse, ApiError>;

    /// Find diocese by ID
    async fn get_diocese_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetDioceseByIdResponse, ApiError>;

    /// Update an existing diocese
    async fn update_diocese(
        &self,
        id: uuid::Uuid,
        diocese: models::Diocese,
        context: &C) -> Result<UpdateDioceseResponse, ApiError>;

    /// Add new parish
    async fn add_parish(
        &self,
        parish: Option<models::Parish>,
        context: &C) -> Result<AddParishResponse, ApiError>;

    /// Delete parish by ID
    async fn delete_parish(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteParishResponse, ApiError>;

    /// Get all parishes
    async fn get_all_parishes(
        &self,
        diocese: Option<models::Diocese>,
        name: Option<String>,
        offset: Option<uuid::Uuid>,
        count: Option<i32>,
        context: &C) -> Result<GetAllParishesResponse, ApiError>;

    /// Find parish by ID
    async fn get_parish_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetParishByIdResponse, ApiError>;

    /// Update an existing parish
    async fn update_parish(
        &self,
        id: uuid::Uuid,
        parish: Option<models::Parish>,
        context: &C) -> Result<UpdateParishResponse, ApiError>;

}

/// API where `Context` isn't passed on every API call
#[async_trait]
pub trait ApiNoContext<C: Send + Sync> {

    fn poll_ready(&self, _cx: &mut Context) -> Poll<Result<(), Box<dyn Error + Send + Sync + 'static>>>;

    fn context(&self) -> &C;

    /// Add new deanery
    async fn add_deanery(
        &self,
        deanery: models::Deanery,
        ) -> Result<AddDeaneryResponse, ApiError>;

    /// Delete deanery by ID
    async fn delete_deanery(
        &self,
        id: uuid::Uuid,
        ) -> Result<DeleteDeaneryResponse, ApiError>;

    /// Get all deaneries
    async fn get_all_deaneries(
        &self,
        province: Option<models::EcclesiasticalProvince>,
        name: Option<String>,
        ) -> Result<GetAllDeaneriesResponse, ApiError>;

    /// Find deanery by ID
    async fn get_deanery_by_id(
        &self,
        id: uuid::Uuid,
        ) -> Result<GetDeaneryByIdResponse, ApiError>;

    /// Update an existing deanery
    async fn update_deanery(
        &self,
        id: uuid::Uuid,
        deanery: models::Deanery,
        ) -> Result<UpdateDeaneryResponse, ApiError>;

    /// Add new diocese
    async fn add_diocese(
        &self,
        diocese: models::Diocese,
        ) -> Result<AddDioceseResponse, ApiError>;

    /// Delete diocese by ID
    async fn delete_diocese(
        &self,
        id: uuid::Uuid,
        ) -> Result<DeleteDioceseResponse, ApiError>;

    /// Get all dioceses
    async fn get_all_dioceses(
        &self,
        province: Option<models::EcclesiasticalProvince>,
        name: Option<String>,
        ) -> Result<GetAllDiocesesResponse, ApiError>;

    /// Find diocese by ID
    async fn get_diocese_by_id(
        &self,
        id: uuid::Uuid,
        ) -> Result<GetDioceseByIdResponse, ApiError>;

    /// Update an existing diocese
    async fn update_diocese(
        &self,
        id: uuid::Uuid,
        diocese: models::Diocese,
        ) -> Result<UpdateDioceseResponse, ApiError>;

    /// Add new parish
    async fn add_parish(
        &self,
        parish: Option<models::Parish>,
        ) -> Result<AddParishResponse, ApiError>;

    /// Delete parish by ID
    async fn delete_parish(
        &self,
        id: uuid::Uuid,
        ) -> Result<DeleteParishResponse, ApiError>;

    /// Get all parishes
    async fn get_all_parishes(
        &self,
        diocese: Option<models::Diocese>,
        name: Option<String>,
        offset: Option<uuid::Uuid>,
        count: Option<i32>,
        ) -> Result<GetAllParishesResponse, ApiError>;

    /// Find parish by ID
    async fn get_parish_by_id(
        &self,
        id: uuid::Uuid,
        ) -> Result<GetParishByIdResponse, ApiError>;

    /// Update an existing parish
    async fn update_parish(
        &self,
        id: uuid::Uuid,
        parish: Option<models::Parish>,
        ) -> Result<UpdateParishResponse, ApiError>;

}

/// Trait to extend an API to make it easy to bind it to a context.
pub trait ContextWrapperExt<C: Send + Sync> where Self: Sized
{
    /// Binds this API to a context.
    fn with_context(self: Self, context: C) -> ContextWrapper<Self, C>;
}

impl<T: Api<C> + Send + Sync, C: Clone + Send + Sync> ContextWrapperExt<C> for T {
    fn with_context(self: T, context: C) -> ContextWrapper<T, C> {
         ContextWrapper::<T, C>::new(self, context)
    }
}

#[async_trait]
impl<T: Api<C> + Send + Sync, C: Clone + Send + Sync> ApiNoContext<C> for ContextWrapper<T, C> {
    fn poll_ready(&self, cx: &mut Context) -> Poll<Result<(), ServiceError>> {
        self.api().poll_ready(cx)
    }

    fn context(&self) -> &C {
        ContextWrapper::context(self)
    }

    /// Add new deanery
    async fn add_deanery(
        &self,
        deanery: models::Deanery,
        ) -> Result<AddDeaneryResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().add_deanery(deanery, &context).await
    }

    /// Delete deanery by ID
    async fn delete_deanery(
        &self,
        id: uuid::Uuid,
        ) -> Result<DeleteDeaneryResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().delete_deanery(id, &context).await
    }

    /// Get all deaneries
    async fn get_all_deaneries(
        &self,
        province: Option<models::EcclesiasticalProvince>,
        name: Option<String>,
        ) -> Result<GetAllDeaneriesResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().get_all_deaneries(province, name, &context).await
    }

    /// Find deanery by ID
    async fn get_deanery_by_id(
        &self,
        id: uuid::Uuid,
        ) -> Result<GetDeaneryByIdResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().get_deanery_by_id(id, &context).await
    }

    /// Update an existing deanery
    async fn update_deanery(
        &self,
        id: uuid::Uuid,
        deanery: models::Deanery,
        ) -> Result<UpdateDeaneryResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().update_deanery(id, deanery, &context).await
    }

    /// Add new diocese
    async fn add_diocese(
        &self,
        diocese: models::Diocese,
        ) -> Result<AddDioceseResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().add_diocese(diocese, &context).await
    }

    /// Delete diocese by ID
    async fn delete_diocese(
        &self,
        id: uuid::Uuid,
        ) -> Result<DeleteDioceseResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().delete_diocese(id, &context).await
    }

    /// Get all dioceses
    async fn get_all_dioceses(
        &self,
        province: Option<models::EcclesiasticalProvince>,
        name: Option<String>,
        ) -> Result<GetAllDiocesesResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().get_all_dioceses(province, name, &context).await
    }

    /// Find diocese by ID
    async fn get_diocese_by_id(
        &self,
        id: uuid::Uuid,
        ) -> Result<GetDioceseByIdResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().get_diocese_by_id(id, &context).await
    }

    /// Update an existing diocese
    async fn update_diocese(
        &self,
        id: uuid::Uuid,
        diocese: models::Diocese,
        ) -> Result<UpdateDioceseResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().update_diocese(id, diocese, &context).await
    }

    /// Add new parish
    async fn add_parish(
        &self,
        parish: Option<models::Parish>,
        ) -> Result<AddParishResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().add_parish(parish, &context).await
    }

    /// Delete parish by ID
    async fn delete_parish(
        &self,
        id: uuid::Uuid,
        ) -> Result<DeleteParishResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().delete_parish(id, &context).await
    }

    /// Get all parishes
    async fn get_all_parishes(
        &self,
        diocese: Option<models::Diocese>,
        name: Option<String>,
        offset: Option<uuid::Uuid>,
        count: Option<i32>,
        ) -> Result<GetAllParishesResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().get_all_parishes(diocese, name, offset, count, &context).await
    }

    /// Find parish by ID
    async fn get_parish_by_id(
        &self,
        id: uuid::Uuid,
        ) -> Result<GetParishByIdResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().get_parish_by_id(id, &context).await
    }

    /// Update an existing parish
    async fn update_parish(
        &self,
        id: uuid::Uuid,
        parish: Option<models::Parish>,
        ) -> Result<UpdateParishResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().update_parish(id, parish, &context).await
    }

}


#[cfg(feature = "client")]
pub mod client;

// Re-export Client as a top-level name
#[cfg(feature = "client")]
pub use client::Client;

#[cfg(feature = "server")]
pub mod server;

// Re-export router() as a top-level name
#[cfg(feature = "server")]
pub use self::server::Service;

#[cfg(feature = "server")]
pub mod context;

pub mod models;

#[cfg(any(feature = "client", feature = "server"))]
pub(crate) mod header;
