#![allow(missing_docs, trivial_casts, unused_variables, unused_mut, unused_imports, unused_extern_crates, non_camel_case_types)]

use async_trait::async_trait;
use futures::Stream;
use std::error::Error;
use std::task::{Poll, Context};
use swagger::{ApiError, ContextWrapper};
use serde::{Serialize, Deserialize};

type ServiceError = Box<dyn Error + Send + Sync + 'static>;

pub const BASE_PATH: &'static str = "";
pub const API_VERSION: &'static str = "0.0.1";

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum AddProgramResponse {
    /// successful operation
    SuccessfulOperation
    (models::Program)
    ,
    /// Invalid input
    InvalidInput
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum DeleteProgramResponse {
    /// Successful operation
    SuccessfulOperation
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// program not found
    ProgramNotFound
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum UpdateProgramResponse {
    /// Successful operation
    SuccessfulOperation
    (models::Program)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// program not found
    ProgramNotFound
    ,
    /// Validation exception
    ValidationException
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum GetProgramByIdResponse {
    /// Successful operation
    SuccessfulOperation
    (models::Program)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// program not found
    ProgramNotFound
}

#[derive(Debug, PartialEq, Serialize, Deserialize)]
#[must_use]
pub enum GetProgramsResponse {
    /// Successful operation
    SuccessfulOperation
    (models::ProgramViewCollection)
    ,
    /// Invalid ID supplied
    InvalidIDSupplied
    ,
    /// program not found
    ProgramNotFound
}

/// API
#[async_trait]
pub trait Api<C: Send + Sync> {
    fn poll_ready(&self, _cx: &mut Context) -> Poll<Result<(), Box<dyn Error + Send + Sync + 'static>>> {
        Poll::Ready(Ok(()))
    }

    /// Add new program
    async fn add_program(
        &self,
        program: models::Program,
        context: &C) -> Result<AddProgramResponse, ApiError>;

    /// Deletes a program
    async fn delete_program(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<DeleteProgramResponse, ApiError>;

    /// Update an existing program
    async fn update_program(
        &self,
        id: uuid::Uuid,
        program: models::Program,
        context: &C) -> Result<UpdateProgramResponse, ApiError>;

    /// Find program by ID
    async fn get_program_by_id(
        &self,
        id: uuid::Uuid,
        context: &C) -> Result<GetProgramByIdResponse, ApiError>;

    /// Get all programs
    async fn get_programs(
        &self,
        name: Option<String>,
        email: Option<String>,
        phone: Option<String>,
        undergraduate_school: Option<String>,
        date_of_birth: Option<chrono::DateTime::<chrono::Utc>>,
        place_of_birth: Option<String>,
        polity_name: Option<String>,
        specialism: Option<String>,
        sorts: Option<&Vec<models::ProgramSortCriteria>>,
        offset: Option<i32>,
        count: Option<i32>,
        context: &C) -> Result<GetProgramsResponse, ApiError>;

}

/// API where `Context` isn't passed on every API call
#[async_trait]
pub trait ApiNoContext<C: Send + Sync> {

    fn poll_ready(&self, _cx: &mut Context) -> Poll<Result<(), Box<dyn Error + Send + Sync + 'static>>>;

    fn context(&self) -> &C;

    /// Add new program
    async fn add_program(
        &self,
        program: models::Program,
        ) -> Result<AddProgramResponse, ApiError>;

    /// Deletes a program
    async fn delete_program(
        &self,
        id: uuid::Uuid,
        ) -> Result<DeleteProgramResponse, ApiError>;

    /// Update an existing program
    async fn update_program(
        &self,
        id: uuid::Uuid,
        program: models::Program,
        ) -> Result<UpdateProgramResponse, ApiError>;

    /// Find program by ID
    async fn get_program_by_id(
        &self,
        id: uuid::Uuid,
        ) -> Result<GetProgramByIdResponse, ApiError>;

    /// Get all programs
    async fn get_programs(
        &self,
        name: Option<String>,
        email: Option<String>,
        phone: Option<String>,
        undergraduate_school: Option<String>,
        date_of_birth: Option<chrono::DateTime::<chrono::Utc>>,
        place_of_birth: Option<String>,
        polity_name: Option<String>,
        specialism: Option<String>,
        sorts: Option<&Vec<models::ProgramSortCriteria>>,
        offset: Option<i32>,
        count: Option<i32>,
        ) -> Result<GetProgramsResponse, ApiError>;

}

/// Trait to extend an API to make it easy to bind it to a context.
pub trait ContextWrapperExt<C: Send + Sync> where Self: Sized
{
    /// Binds this API to a context.
    fn with_context(self: Self, context: C) -> ContextWrapper<Self, C>;
}

impl<T: Api<C> + Send + Sync, C: Clone + Send + Sync> ContextWrapperExt<C> for T {
    fn with_context(self: T, context: C) -> ContextWrapper<T, C> {
         ContextWrapper::<T, C>::new(self, context)
    }
}

#[async_trait]
impl<T: Api<C> + Send + Sync, C: Clone + Send + Sync> ApiNoContext<C> for ContextWrapper<T, C> {
    fn poll_ready(&self, cx: &mut Context) -> Poll<Result<(), ServiceError>> {
        self.api().poll_ready(cx)
    }

    fn context(&self) -> &C {
        ContextWrapper::context(self)
    }

    /// Add new program
    async fn add_program(
        &self,
        program: models::Program,
        ) -> Result<AddProgramResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().add_program(program, &context).await
    }

    /// Deletes a program
    async fn delete_program(
        &self,
        id: uuid::Uuid,
        ) -> Result<DeleteProgramResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().delete_program(id, &context).await
    }

    /// Update an existing program
    async fn update_program(
        &self,
        id: uuid::Uuid,
        program: models::Program,
        ) -> Result<UpdateProgramResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().update_program(id, program, &context).await
    }

    /// Find program by ID
    async fn get_program_by_id(
        &self,
        id: uuid::Uuid,
        ) -> Result<GetProgramByIdResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().get_program_by_id(id, &context).await
    }

    /// Get all programs
    async fn get_programs(
        &self,
        name: Option<String>,
        email: Option<String>,
        phone: Option<String>,
        undergraduate_school: Option<String>,
        date_of_birth: Option<chrono::DateTime::<chrono::Utc>>,
        place_of_birth: Option<String>,
        polity_name: Option<String>,
        specialism: Option<String>,
        sorts: Option<&Vec<models::ProgramSortCriteria>>,
        offset: Option<i32>,
        count: Option<i32>,
        ) -> Result<GetProgramsResponse, ApiError>
    {
        let context = self.context().clone();
        self.api().get_programs(name, email, phone, undergraduate_school, date_of_birth, place_of_birth, polity_name, specialism, sorts, offset, count, &context).await
    }

}


#[cfg(feature = "client")]
pub mod client;

// Re-export Client as a top-level name
#[cfg(feature = "client")]
pub use client::Client;

#[cfg(feature = "server")]
pub mod server;

// Re-export router() as a top-level name
#[cfg(feature = "server")]
pub use self::server::Service;

#[cfg(feature = "server")]
pub mod context;

pub mod models;

#[cfg(any(feature = "client", feature = "server"))]
pub(crate) mod header;
