//! This module exports a generic invariant of http uri that has query

use crate::{define_generic_invariant, HttpUri};

/// An enum of requirements for a http uri to be with query.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum HttpUriWithQueryRequirement {
    #[strum(serialize = "Uri must have query")]
    UriMustHaveQuery,
}

fn detect_any_violated_requirement(uri: &HttpUri<'_>) -> Option<HttpUriWithQueryRequirement> {
    // Ensure uri has query
    if !uri.has_query() {
        return Some(HttpUriWithQueryRequirement::UriMustHaveQuery);
    }
    None
}

define_generic_invariant!(
    /// A http uri, that has query part.
    HttpUriWithQuery,
    HttpUri,
    'uri,
    HttpUriWithQueryRequirement,
    HttpUriWithQueryRequirementViolation,
    detect_any_violated_requirement,
);

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_uri_with_query(
        uri_str: &'static str,
    ) -> Result<
        HttpUriWithQuery<'static, HttpUri<'static>>,
        HttpUriWithQueryRequirementViolation<'static, HttpUri<'static>>,
    > {
        let http_uri = HttpUri::try_from(uri_str).unwrap();
        Cow::<'static, HttpUri<'static>>::Owned(http_uri).try_into()
    }

    #[rstest]
    #[case("http://example.org/path/to/a#abc")]
    #[case("http://example.org/")]
    #[case("https://example.org/path/to/a/")]
    #[trace]
    fn http_uri_with_out_query_will_be_rejected(#[case] uri_str: &'static str) {
        HttpUriWithQueryRequirement::UriMustHaveQuery
            .assert_violation(assert_err!(try_uri_with_query(uri_str)))
    }

    #[rstest]
    #[case("http://example.org?")]
    #[case("http://example.org/?")]
    #[case("http://localhost/?q1")]
    #[case("http://example.org/path/to/a?q2")]
    #[case("https://example.org/path/to/a/?abc#f1")]
    #[trace]
    fn http_uri_with_query_will_be_accepted(#[case] uri_str: &'static str) {
        assert_ok!(try_uri_with_query(uri_str));
    }
}
