//! This module exports a generic invariant of http uri that doesn't has trailing slash in it's path

use crate::{define_generic_invariant, HttpUri};

/// An enum of requirements for a http uri to be sans trailing slash in it's path.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum HttpUriSansTslashRequirement {
    #[strum(serialize = "Uri must not have trailing slash in it's path")]
    UriMustNotHaveTrailingSlashInPath,
}

fn detect_any_violated_requirement(uri: &HttpUri<'_>) -> Option<HttpUriSansTslashRequirement> {
    // Ensure uri path has no trailing slash
    if let Some(last_segment) = uri.path().segments().last() {
        // If path has a trailing slash, it implies it's last segment is empty
        if last_segment.is_empty() {
            return Some(HttpUriSansTslashRequirement::UriMustNotHaveTrailingSlashInPath);
        }
    }
    None
}

define_generic_invariant!(
    /// A http uri, that doesn't has trailing slash in it's path.
    HttpUriSansTslash,
    HttpUri,
    'uri,
    HttpUriSansTslashRequirement,
    HttpUriSansTslashRequirementViolation,
    detect_any_violated_requirement,
);

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_uri_sans_tslash(
        uri_str: &'static str,
    ) -> Result<
        HttpUriSansTslash<'static, HttpUri<'static>>,
        HttpUriSansTslashRequirementViolation<'static, HttpUri<'static>>,
    > {
        let http_uri = HttpUri::try_from(uri_str).unwrap();
        Cow::<'static, HttpUri<'static>>::Owned(http_uri).try_into()
    }

    #[rstest]
    #[case::implicit_root_slash1("http://example.org")]
    #[case::implicit_root_slash2("http://example.org#")]
    #[case::implicit_root_slash3("http://example.org?")]
    #[case("http://example.org/")]
    #[case("http://localhost/?a#f3")]
    #[case("https://example.org/path/to/a/?abc#f3")]
    #[trace]
    fn http_uri_with_tslash_will_be_rejected(#[case] uri_str: &'static str) {
        HttpUriSansTslashRequirement::UriMustNotHaveTrailingSlashInPath
            .assert_violation(assert_err!(try_uri_sans_tslash(uri_str)))
    }

    #[rstest]
    #[case("http://example.org/path/to/a")]
    #[case("https://example.org/path/to/b?c/")]
    #[case("https://example.org/path/to/b#")]
    #[trace]
    fn http_uri_with_out_tslash_will_be_accepted(#[case] uri_str: &'static str) {
        assert_ok!(try_uri_sans_tslash(uri_str));
    }
}
