//! This module exports a generic invariant of http uri that is in normalized form

use std::{borrow::Cow, marker::PhantomData};

use crate::{
    base::{_ext::uri::URIX, tdd::invariant::TypeInvariant},
    define_generic_invariant, HttpUri,
};

/// An enum of requirements for a http uri to be normal.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum NormalHttpUriRequirement {
    #[strum(serialize = "Uri must be normal per rfc 3986")]
    UriMustBeNormalPerRfc,
    #[strum(serialize = "Uri must not have explicit http(s) default port")]
    UriMustNotHaveExplicitDefaultPort,
    #[strum(serialize = "Uri path must not have non trailing empty segments")]
    UriMustNotHaveNonTrailingEmptySegments,
}

fn detect_any_violated_requirement(uri: &HttpUri<'_>) -> Option<NormalHttpUriRequirement> {
    // Ensure uri is normalized per rfc
    if !uri.is_normalized() {
        return Some(NormalHttpUriRequirement::UriMustBeNormalPerRfc);
    }
    // Ensure http uri has no explicit default port
    if uri.has_explicit_default_port() {
        return Some(NormalHttpUriRequirement::UriMustNotHaveExplicitDefaultPort);
    }
    // Ensure uri path has no non trailing empty segments
    if uri.has_non_trailing_empty_segments() {
        return Some(NormalHttpUriRequirement::UriMustNotHaveNonTrailingEmptySegments);
    }
    None
}

define_generic_invariant!(
    /// A http uri, that is in normal form.
    /// By normal it entails:
    ///
    /// 1. Valid http uri
    /// 2. chars in unreserved charset are not pct-encoded.in path or query
    /// 3. All pct-encoded octets are in upper case
    /// 4. There are no dot-segments in path
    /// 5. No explicit default port;
    /// 6. Except if trailing segment, no other path segment is empty
    NormalHttpUri,
    HttpUri,
    'uri,
    NormalHttpUriRequirement,
    NormalHttpUriRequirementViolation,
    detect_any_violated_requirement,
);

/// A marker trait that says, given http uri type invariant will not corrupt it's invariant properties on http-uri-normalization.
///
/// # Safety
/// This trait must not be implemented if by normalization of url, the invariant properties will not hold.
pub unsafe trait HttpUriNormalizationOrthogonal<'uri>: TypeInvariant<HttpUri<'uri>> {}

impl<'uri, HUI> NormalHttpUri<'uri, HUI>
where
    HUI: HttpUriNormalizationOrthogonal<'uri>
        + Clone
        + PartialEq
        + TryFrom<Cow<'uri, HttpUri<'uri>>>,
    <HUI as TryFrom<Cow<'uri, HttpUri<'uri>>>>::Error: std::fmt::Debug,
{
    /// Create `HttpUriNormal` instance from given http uri invariant.
    /// If it is not normal, it will be normalized first
    pub fn normalized_from(uri_invariant: HUI) -> Self {
        match Self::try_from(Cow::Owned(uri_invariant)) {
            // If given http uri is already normal
            Ok(normal_uri) => normal_uri,
            // If given http uri is not normal
            Err(e) => {
                let mut uri: HttpUri<'uri> = e.value.into_owned().into();
                // Normalize inner uri per rfc
                uri.normalize_inner_uri();
                // Remove explicit default port
                uri.remove_explicit_default_port();
                // Remove non-trailing empty segments
                uri.remove_non_trailing_empty_segments();

                Self {
                    inner_invariant: Cow::<'uri, HttpUri<'uri>>::Owned(uri).try_into().unwrap(),
                    _phantom: PhantomData,
                }
            }
        }
    }
}

/// Tests for creation of [`NormalHttpUri`] from [`HttpUri`]
#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_normal_http_uri(
        uri_str: &'static str,
    ) -> Result<
        NormalHttpUri<'static, HttpUri<'static>>,
        NormalHttpUriRequirementViolation<'static, HttpUri<'static>>,
    > {
        let http_uri = HttpUri::try_from(uri_str).unwrap();
        Cow::<'static, HttpUri<'static>>::Owned(http_uri).try_into()
    }

    #[rstest]
    #[case::a("http://pod1.example.org/kosala/%61yodhya")]
    #[case::a_q("http://pod1.example.org/kosala?%61yodhya")]
    #[case::a_cap("http://pod1.example.org/kosala/%41yodhya")]
    #[case::dig_1("http://pod1.example.org/kosala/raghu%31")]
    #[case::hyphen("http://pod1.example.org/rama%2Drajya")]
    #[case::tilde("http://pod1.example.org/rama%7Esita")]
    #[case::tilde_q("http://pod1.example.org/?rama%7Esita")]
    #[case::period("http://pod1.example.org/kosala%2Eayodhya")]
    #[case::period_2("http://pod1.example.org/path/to/%2E%2E/c")]
    #[case::underscore("http://pod1.example.org/rama%5Flakshmana")]
    #[trace]
    fn http_uri_un_normal_with_un_reserved_char_encoded_will_be_rejected(
        #[case] uri_str: &'static str,
    ) {
        NormalHttpUriRequirement::UriMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_http_uri(uri_str)));
    }

    #[rstest]
    #[case("http://pod1.example.org/rama%3ddharma")]
    #[case("http://pod1.example.org/?rama%3ddharma")]
    #[case("http://pod1.example.org/rama%2blakshmana")]
    #[case("http://pod1.example.org/rama%2clakshmana")]
    #[case("http://pod1.example.org/%e0%A4%b0%E0%A4%BE%E0%A4%AE")]
    #[case("http://pod1.example.org/the?%e0%A4%b0%E0%A4%BE%E0%A4%AE")]
    #[case("http://pod1.example.org/kosala%2fayodhya")]
    #[trace]
    fn http_uri_un_normal_with_lowercase_pct_encoded_will_be_rejected(
        #[case] uri_str: &'static str,
    ) {
        NormalHttpUriRequirement::UriMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_http_uri(uri_str)));
    }

    #[rstest]
    // #[case("HTTP://pod1.example.org/rama")] ; scheme is auto normalized by parser
    #[case("http://pod1.EXAMPLE.org/")]
    #[case("http://pod1.example.OEG/rama%2blakshmana")]
    #[case("http://pod1.ExAmple.org/rama%2clakshmana")]
    #[trace]
    fn http_uri_un_normal_with_uppercase_origin_char_will_be_rejected(
        #[case] uri_str: &'static str,
    ) {
        NormalHttpUriRequirement::UriMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_http_uri(uri_str)));
    }

    #[rstest]
    #[case("http://pod1.example.org/.")]
    #[case("http://pod1.example.org/..")]
    #[case("http://pod1.example.org/./path/to/a")]
    #[case("http://pod1.example.org/../a")]
    #[case("http://pod1.example.org/path/to/a/../b/")]
    #[case("http://pod1.example.org/path/to/c/.")]
    #[case("http://pod1.example.org/path/to/c/./")]
    #[case("http://pod1.example.org/path/to/c/..")]
    #[case("http://pod1.example.org/path/to/c/../")]
    #[trace]
    fn http_uri_un_normal_with_dot_segments_will_be_rejected(#[case] uri_str: &'static str) {
        NormalHttpUriRequirement::UriMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_http_uri(uri_str)));
    }

    #[rstest]
    #[case::http_80("http://pod1.example.org:80/path/to/a")]
    #[case::https_443("https://pod1.example.org:443/a?b#cde")]
    #[case::http_localhost_80("http://localhost:80")]
    #[trace]
    fn http_uri_with_explicit_default_port_will_be_rejected(#[case] uri_str: &'static str) {
        NormalHttpUriRequirement::UriMustNotHaveExplicitDefaultPort
            .assert_violation(assert_err!(try_normal_http_uri(uri_str)));
    }

    #[rstest]
    #[case("http://pod1.example.org/path/to//a")]
    #[case("http://pod1.example.org/path/to//")]
    #[trace]
    fn http_uri_with_non_trailing_empty_segment_will_be_rejected(#[case] uri_str: &'static str) {
        NormalHttpUriRequirement::UriMustNotHaveNonTrailingEmptySegments
            .assert_violation(assert_err!(try_normal_http_uri(uri_str)));
    }

    #[rstest]
    #[case::localhost_authority("http://localhost/a/b/c")]
    #[case::ip_authority("http://127.0.0.1/a/b/c")]
    #[case::explicit_port1("http://pod1.example.org:8000/a/b/c")]
    #[case::explicit_port2("http://pod1.example.org:8443/")]
    #[case::explicit_port2("http://pod1.example.org:8443/")]
    #[case::non_relative_period("http://pod1.example.org/a.acl")]
    #[case("http://pod1.example.org/kosala/ayodhya")]
    #[case("http://pod1.example.org/path/to/container/")]
    #[case::un_reserved_unencoded_1("http://pod1.example.org/path/to/container/.acl")]
    #[case::un_reserved_unencoded_1_q("http://pod1.example.org/path/to/container/?.acl")]
    #[case::un_reserved_unencoded_2("http://pod1.example.org/path/to/container/~acl")]
    #[case::un_reserved_unencoded_3("http://pod1.example.org/path/to/container/_acl")]
    #[case::un_reserved_unencoded_4("http://pod1.example.org/path/to/container/-acl")]
    #[case::sub_delim_unencoded_1("http://pod1.example.org/path/to/container/$acl")]
    #[case::sub_delim_unencoded_2("http://pod1.example.org/rama=dharma")]
    #[case::sub_delim_unencoded_3("http://pod1.example.org/rama+lakshmana")]
    #[case::sub_delim_unencoded_4("http://pod1.example.org/rama,lakshmana")]
    #[case::sub_delim_unencoded_5("http://pod1.example.org/rama&lakshmana")]
    #[case::sub_delim_encoded_1("http://pod1.example.org/path/to/container/%24acl")]
    #[case::sub_delim_encoded_2("http://pod1.example.org/rama%3Ddharma")]
    #[case::sub_delim_encoded_3("http://pod1.example.org/rama%2Blakshmana")]
    #[case::sub_delim_encoded_4("http://pod1.example.org/rama%2Clakshmana")]
    #[case::sub_delim_encoded_5("http://pod1.example.org/rama%26lakshmana")]
    #[case::excepted_gen_delim_unencoded_1("http://pod1.example.org/a:b")]
    #[case::expected_gen_delim_unencoded_2("http://pod1.example.org/a@b")]
    #[case::excepted_gen_delim_encoded_1("http://pod1.example.org/a%3Ab")]
    #[case::expected_gen_delim_encoded_2("http://pod1.example.org/a%40b")]
    #[case::gen_delim_encoded_1("http://pod1.example.org/a/b%2Fc")]
    #[case::gen_delim_encoded_2("http://pod1.example.org/a/b%3Fc")]
    #[case::gen_delim_encoded_3("http://pod1.example.org/a/b%23c")]
    #[case::gen_delim_encoded_4("http://pod1.example.org/a/b%5B%5Dc")]
    #[case::non_ascii_pct_encoded("http://pod1.example.org/ramayana/%E0%A4%B0%E0%A4%BE%E0%A4%AE")]
    #[case::non_ascii_pct_encoded(
        "http://pod1.example.org/%E0%B0%85%E0%B0%AF%E0%B1%8B%E0%B0%A7%E0%B1%8D%E0%B0%AF"
    )]
    #[trace]
    fn normalized_http_uri_will_be_accepted(#[case] uri_str: &'static str) {
        assert_ok!(try_normal_http_uri(uri_str));
    }
}

/// Tests for normalization of [`HttpUri`] to [`NormalHttpUri`].
#[cfg(test)]
pub mod test_normalized_from {
    use rstest::rstest;

    use super::*;

    fn assert_correct_normalization(source_uri_str: &'static str, expected_uri_str: &'static str) {
        let http_uri = HttpUri::try_from(source_uri_str).unwrap();
        let normal_http_uri = NormalHttpUri::normalized_from(http_uri);
        assert_eq!(normal_http_uri.as_ref(), expected_uri_str);
    }

    #[rstest]
    #[case::a(
        "http://pod1.example.org/kosala/%61yodhya",
        "http://pod1.example.org/kosala/ayodhya"
    )]
    #[case::a_q(
        "http://pod1.example.org/kosala?%61yodhya",
        "http://pod1.example.org/kosala?ayodhya"
    )]
    #[case::a_cap(
        "http://pod1.example.org/kosala/%41yodhya",
        "http://pod1.example.org/kosala/Ayodhya"
    )]
    #[case::dig_1(
        "http://pod1.example.org/kosala/raghu%31",
        "http://pod1.example.org/kosala/raghu1"
    )]
    #[case::hyphen(
        "http://pod1.example.org/rama%2Drajya",
        "http://pod1.example.org/rama-rajya"
    )]
    #[case::tilde(
        "http://pod1.example.org/rama%7Esita",
        "http://pod1.example.org/rama~sita"
    )]
    #[case::period(
        "http://pod1.example.org/kosala%2Eayodhya",
        "http://pod1.example.org/kosala.ayodhya"
    )]
    #[case::period_2(
        "http://pod1.example.org/path/to/%2E%2E/c",
        "http://pod1.example.org/path/c"
    )]
    #[case::underscore(
        "http://pod1.example.org/rama%5Flakshmana",
        "http://pod1.example.org/rama_lakshmana"
    )]
    #[trace]
    fn unreserved_char_encoding_will_be_normalized_correctly(
        #[case] source_uri_str: &'static str,
        #[case] expected_uri_str: &'static str,
    ) {
        assert_correct_normalization(source_uri_str, expected_uri_str);
    }

    #[rstest]
    #[case(
        "http://pod1.example.org/rama%3ddharma",
        "http://pod1.example.org/rama%3Ddharma"
    )]
    #[case(
        "http://pod1.example.org/?rama%3ddharma",
        "http://pod1.example.org/?rama%3Ddharma"
    )]
    #[case(
        "http://pod1.example.org/rama%2blakshmana",
        "http://pod1.example.org/rama%2Blakshmana"
    )]
    #[case(
        "http://pod1.example.org/rama%2clakshmana",
        "http://pod1.example.org/rama%2Clakshmana"
    )]
    #[case(
        "http://pod1.example.org/%e0%A4%b0%E0%A4%BE%E0%A4%AE",
        "http://pod1.example.org/%E0%A4%B0%E0%A4%BE%E0%A4%AE"
    )]
    #[case(
        "http://pod1.example.org/kosala%2fayodhya",
        "http://pod1.example.org/kosala%2Fayodhya"
    )]
    #[trace]
    fn pct_encoded_octets_case_will_be_normalized_correctly(
        #[case] source_uri_str: &'static str,
        #[case] expected_uri_str: &'static str,
    ) {
        assert_correct_normalization(source_uri_str, expected_uri_str);
    }

    #[rstest]
    #[case("http://pod1.example.org/.", "http://pod1.example.org/")]
    #[case("http://pod1.example.org/..", "http://pod1.example.org/")]
    #[case(
        "http://pod1.example.org/./path/to/a",
        "http://pod1.example.org/path/to/a"
    )]
    #[case("http://pod1.example.org/../a", "http://pod1.example.org/a")]
    #[case("http://pod1.example.org/./a:b", "http://pod1.example.org/a:b")]
    #[case(
        "http://pod1.example.org/path/to/a/../b/",
        "http://pod1.example.org/path/to/b/"
    )]
    #[case(
        "http://pod1.example.org/path/to/c/.",
        "http://pod1.example.org/path/to/c/"
    )]
    #[case(
        "http://pod1.example.org/path/to/c/./",
        "http://pod1.example.org/path/to/c/"
    )]
    #[case(
        "http://pod1.example.org/path/to/c/..",
        "http://pod1.example.org/path/to/"
    )]
    #[case(
        "http://pod1.example.org/path/to/c/../",
        "http://pod1.example.org/path/to/"
    )]
    #[case::pct_encoded_dot_segments(
        "http://pod1.example.org/path/to/%2E%2E/c",
        "http://pod1.example.org/path/c"
    )]
    #[trace]
    fn dot_segments_will_be_normalized_correctly(
        #[case] source_uri_str: &'static str,
        #[case] expected_uri_str: &'static str,
    ) {
        assert_correct_normalization(source_uri_str, expected_uri_str);
    }

    #[rstest]
    #[case::http("http://pod1.example.org/a/b?q", "http://pod1.example.org/a/b?q")]
    #[case::http_80("http://pod1.example.org:80/a/b?q", "http://pod1.example.org/a/b?q")]
    #[case::http_8000(
        "http://pod1.example.org:8000/a/b?q",
        "http://pod1.example.org:8000/a/b?q"
    )]
    #[case::https("https://pod1.example.org/a/b?q", "https://pod1.example.org/a/b?q")]
    #[case::https_443("https://pod1.example.org:443/a/b?q", "https://pod1.example.org/a/b?q")]
    #[case::https_743(
        "https://pod1.example.org:743/a/b?q",
        "https://pod1.example.org:743/a/b?q"
    )]
    #[trace]
    fn port_will_be_normalized_correctly(
        #[case] source_uri_str: &'static str,
        #[case] expected_uri_str: &'static str,
    ) {
        assert_correct_normalization(source_uri_str, expected_uri_str);
    }

    #[rstest]
    #[case(
        "http://pod1.example.org/path/to//a",
        "http://pod1.example.org/path/to/a"
    )]
    #[case(
        "http://pod1.example.org/path/to//",
        "http://pod1.example.org/path/to/"
    )]
    #[trace]
    fn non_trailing_empty_segments_will_be_normalized_correctly(
        #[case] source_uri_str: &'static str,
        #[case] expected_uri_str: &'static str,
    ) {
        assert_correct_normalization(source_uri_str, expected_uri_str);
    }
}
