use std::borrow::Cow;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    invariant::sans_component::query::{HttpUriSansQuery, HttpUriSansQueryRequirement},
    HttpUri,
};

use super::absolute_normalized::{AbsoluteNormalizedHttpUri, AbsoluteNormalizedHttpUriRequirement};

/// Concrete type for an http uri invariant, that is guaranteed to be normalized, and is absolute (i.e. no fragment), and no query
pub type HierarchicalNormalizedHttpUri<'uri> =
    HttpUriSansQuery<'uri, AbsoluteNormalizedHttpUri<'uri>>;

/// An enum of requirements for a http uri to be hierarchical-normalized.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum HierarchicalNormalizedHttpUriRequirement {
    #[strum(serialize = "HttpUri must be absolute and normalized")]
    HttpUriMustBeAbsoluteNormalized(AbsoluteNormalizedHttpUriRequirement),
    #[strum(serialize = "HttpUri must be sans query")]
    HttpUriMustBeSansQuery(HttpUriSansQueryRequirement),
}

impl From<AbsoluteNormalizedHttpUriRequirement> for HierarchicalNormalizedHttpUriRequirement {
    fn from(r: AbsoluteNormalizedHttpUriRequirement) -> Self {
        Self::HttpUriMustBeAbsoluteNormalized(r)
    }
}

impl From<HttpUriSansQueryRequirement> for HierarchicalNormalizedHttpUriRequirement {
    fn from(r: HttpUriSansQueryRequirement) -> Self {
        Self::HttpUriMustBeSansQuery(r)
    }
}

impl InvariantRequirement for HierarchicalNormalizedHttpUriRequirement {}

pub type HierarchicalNormalizedHttpUriRequirementViolation<'uri> =
    InvariantRequirementViolation<HttpUri<'uri>, HierarchicalNormalizedHttpUriRequirement>;

impl<'uri> TryFrom<HttpUri<'uri>> for HierarchicalNormalizedHttpUri<'uri> {
    type Error = HierarchicalNormalizedHttpUriRequirementViolation<'uri>;

    fn try_from(http_uri: HttpUri<'uri>) -> Result<Self, Self::Error> {
        HttpUriSansQuery::try_from(Cow::Owned(
            AbsoluteNormalizedHttpUri::try_from(http_uri)
                .map_err(InvariantRequirementViolation::from_inner)?,
        ))
        .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
