use std::borrow::Cow;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    invariant::sans_component::fragment::{HttpUriSansFragment, HttpUriSansFragmentRequirement},
    HttpUri,
};

use super::normalized::{NormalizedHttpUri, NormalizedHttpUriRequirement};

/// Concrete type for an http uri invariant, that is guaranteed to be normalized, and is absolute (i.e. no fragment)
pub type AbsoluteNormalizedHttpUri<'uri> = HttpUriSansFragment<'uri, NormalizedHttpUri<'uri>>;

/// An enum of requirements for a http uri to be absolute-normalized.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum AbsoluteNormalizedHttpUriRequirement {
    #[strum(serialize = "HttpUri must be normalized")]
    HttpUriMustBeNormalized(NormalizedHttpUriRequirement),
    #[strum(serialize = "HttpUri must be sans fragment")]
    HttpUriMustBeSansFragment(HttpUriSansFragmentRequirement),
}

impl From<NormalizedHttpUriRequirement> for AbsoluteNormalizedHttpUriRequirement {
    fn from(r: NormalizedHttpUriRequirement) -> Self {
        Self::HttpUriMustBeNormalized(r)
    }
}

impl From<HttpUriSansFragmentRequirement> for AbsoluteNormalizedHttpUriRequirement {
    fn from(r: HttpUriSansFragmentRequirement) -> Self {
        Self::HttpUriMustBeSansFragment(r)
    }
}

impl InvariantRequirement for AbsoluteNormalizedHttpUriRequirement {}

pub type AbsoluteNormalizedHttpUriRequirementViolation<'uri> =
    InvariantRequirementViolation<HttpUri<'uri>, AbsoluteNormalizedHttpUriRequirement>;

impl<'uri> TryFrom<HttpUri<'uri>> for AbsoluteNormalizedHttpUri<'uri> {
    type Error = AbsoluteNormalizedHttpUriRequirementViolation<'uri>;

    fn try_from(http_uri: HttpUri<'uri>) -> Result<Self, Self::Error> {
        HttpUriSansFragment::try_from(Cow::Owned(
            NormalizedHttpUri::try_from(http_uri)
                .map_err(InvariantRequirementViolation::from_inner)?,
        ))
        .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
