//! This module exports a generic invariant of segment that is not empty

use uriparse::Segment;

use crate::define_generic_invariant;

/// An enum of requirements for a segment to be non empty.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum NonEmptySegmentRequirement {
    #[strum(serialize = "Segment must not be empty")]
    SegmentMustBeNonEmpty,
}

fn detect_any_violated_requirement(segment: &Segment<'_>) -> Option<NonEmptySegmentRequirement> {
    // Ensure segment is not empty
    if segment.is_empty() {
        return Some(NonEmptySegmentRequirement::SegmentMustBeNonEmpty);
    }
    None
}

define_generic_invariant!(
    /// A segment that is not empty
    NonEmptySegment,
    Segment,
    'segment,
    NonEmptySegmentRequirement,
    NonEmptySegmentRequirementViolation,
    detect_any_violated_requirement,
);

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_non_empty_segment(
        segment_str: &'static str,
    ) -> Result<
        NonEmptySegment<'static, Segment<'static>>,
        NonEmptySegmentRequirementViolation<'static, Segment<'static>>,
    > {
        let segment = Segment::try_from(segment_str).unwrap();
        Cow::<'static, Segment<'static>>::Owned(segment).try_into()
    }

    #[rstest]
    #[case("")]
    #[trace]
    fn empty_segment_will_be_rejected(#[case] segment_str: &'static str) {
        NonEmptySegmentRequirement::SegmentMustBeNonEmpty
            .assert_violation(assert_err!(try_non_empty_segment(segment_str)))
    }

    #[rstest]
    #[case("abc")]
    #[case(".")]
    #[case("..")]
    #[case("image.png")]
    #[trace]
    fn non_empty_segment_will_be_accepted(#[case] segment_str: &'static str) {
        assert_ok!(try_non_empty_segment(segment_str));
    }
}
