use std::borrow::Cow;

use uriparse::Segment;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    component::segment::invariant::non_empty::{NonEmptySegment, NonEmptySegmentRequirement},
};

use super::clean::{CleanSegment, CleanSegmentRequirement};

/// Concrete type for a segment invariant, that is guaranteed to be normalized, and is non dot segment, and non empty.
pub type CleanNonEmptySegment<'segment> = NonEmptySegment<'segment, CleanSegment<'segment>>;

/// An enum of requirements for a segment to be clean.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum CleanNonEmptySegmentRequirement {
    #[strum(serialize = "Segment must be clean")]
    SegmentMustBeClean(CleanSegmentRequirement),
    #[strum(serialize = "Segment must be non empty segment")]
    SegmentMustBeNonEmptySegment(NonEmptySegmentRequirement),
}

impl From<CleanSegmentRequirement> for CleanNonEmptySegmentRequirement {
    fn from(r: CleanSegmentRequirement) -> Self {
        Self::SegmentMustBeClean(r)
    }
}

impl From<NonEmptySegmentRequirement> for CleanNonEmptySegmentRequirement {
    fn from(r: NonEmptySegmentRequirement) -> Self {
        Self::SegmentMustBeNonEmptySegment(r)
    }
}

impl InvariantRequirement for CleanNonEmptySegmentRequirement {}

pub type CleanNonEmptySegmentRequirementViolation<'segment> =
    InvariantRequirementViolation<Segment<'segment>, CleanNonEmptySegmentRequirement>;

impl<'segment> TryFrom<Segment<'segment>> for CleanNonEmptySegment<'segment> {
    type Error = CleanNonEmptySegmentRequirementViolation<'segment>;

    fn try_from(segment: Segment<'segment>) -> Result<Self, Self::Error> {
        NonEmptySegment::try_from(Cow::Owned(
            CleanSegment::try_from(segment).map_err(InvariantRequirementViolation::from_inner)?,
        ))
        .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
