//! This module exports a generic invariant of rootless path

use uriparse::Path;

use crate::define_generic_invariant;

/// An enum of requirements for a path to be rootless.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum RootlessPathRequirement {
    #[strum(serialize = "Path must be rootless")]
    PathMustBeRootless,
}

fn detect_any_violated_requirement(path: &Path<'_>) -> Option<RootlessPathRequirement> {
    // Ensure path is rootless
    if !path.is_relative() {
        return Some(RootlessPathRequirement::PathMustBeRootless);
    }
    None
}

define_generic_invariant!(
    /// An invariant of path, that is rootless as per rfc3986
    RootlessPath,
    Path,
    'path,
    RootlessPathRequirement,
    RootlessPathRequirementViolation,
    detect_any_violated_requirement,
);

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_rootless_path(
        path_str: &'static str,
    ) -> Result<
        RootlessPath<'static, Path<'static>>,
        RootlessPathRequirementViolation<'static, Path<'static>>,
    > {
        let path = Path::try_from(path_str).unwrap();
        Cow::<'static, Path<'static>>::Owned(path).try_into()
    }

    #[rstest]
    #[case("/")]
    #[case("/..")]
    #[case("/abc")]
    #[case("/abc/def/")]
    #[trace]
    fn non_rootless_path_will_be_rejected(#[case] path_str: &'static str) {
        RootlessPathRequirement::PathMustBeRootless
            .assert_violation(assert_err!(try_rootless_path(path_str)))
    }

    #[rstest]
    #[case("")]
    #[case(".")]
    #[case("abc")]
    #[case("abc/def")]
    #[trace]
    fn rootless_path_will_be_accepted(#[case] path_str: &'static str) {
        assert_ok!(try_rootless_path(path_str));
    }
}
