use std::borrow::Cow;

use uriparse::Path;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    component::path::invariant::normal::{NormalPath, NormalPathRequirement},
};

/// Concrete type for a path invariant, that is guaranteed to be normalized
pub type NormalizedPath<'path> = NormalPath<'path, Path<'path>>;

/// An enum of requirements for a segment to be normalized.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum NormalizedPathRequirement {
    #[strum(serialize = "Path must be normal")]
    PathMustBeNormal(NormalPathRequirement),
}

impl From<NormalPathRequirement> for NormalizedPathRequirement {
    fn from(r: NormalPathRequirement) -> Self {
        Self::PathMustBeNormal(r)
    }
}

impl InvariantRequirement for NormalizedPathRequirement {}

pub type NormalizedPathRequirementViolation<'path> =
    InvariantRequirementViolation<Path<'path>, NormalizedPathRequirement>;

impl<'path> TryFrom<Path<'path>> for NormalizedPath<'path> {
    type Error = NormalizedPathRequirementViolation<'path>;

    fn try_from(segment: Path<'path>) -> Result<Self, Self::Error> {
        NormalPath::try_from(Cow::Owned(segment))
            .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
