use std::{
    borrow::{Borrow, Cow},
    fmt::Debug,
};

pub mod define;

/// A trait that marks a type as invariant of other base type.
pub trait TypeInvariant<BaseType>: Into<BaseType> + Borrow<BaseType> {}

impl<T> TypeInvariant<T> for T {}

/// A marker type for types, that encodes requirement of a type invariant, from it's base-type
pub trait InvariantRequirement: Debug {}

/// An error of violating invariant requirement
#[derive(Debug, thiserror::Error)]
pub struct InvariantRequirementViolation<V, RQ: InvariantRequirement> {
    pub value: V,
    pub requirement: RQ,
}

pub trait IntoOwned {}

impl<V, RQ> InvariantRequirementViolation<V, RQ>
where
    V: Clone,
    RQ: InvariantRequirement,
{
    pub fn from_inner<IRQ>(e: InvariantRequirementViolation<V, IRQ>) -> Self
    where
        IRQ: InvariantRequirement + Into<RQ>,
    {
        let InvariantRequirementViolation { value, requirement } = e;
        Self {
            value,
            requirement: requirement.into(),
        }
    }

    pub fn from_inner_cow<IV, IRQ>(e: InvariantRequirementViolation<Cow<IV>, IRQ>) -> Self
    where
        IV: TypeInvariant<V> + Clone,
        IRQ: InvariantRequirement + Into<RQ>,
    {
        let InvariantRequirementViolation { value, requirement } = e;
        Self {
            value: value.into_owned().into(),
            requirement: requirement.into(),
        }
    }
}
