//! This crate exports types for that represents http uri and some of it's invariants.
//! All types are wrappers around [`uriparse::URI`] type, and enforces their corresponding invariants
//!
//! 1. [`HttpUri`]: It guarantees uri is valid, http(s), with valid host
//! 2. [`invariant::normal::NormalHttpUri`]: Invariant of [`HttpUri`] that guarantees, the inner uri is a normalized http uri. Check type docs for what it entails to be normal
//!
//! There are few other invariants based on presence/absence of certain components in modules `invariant::with_component::*`, `invariant::sans_component::*`.
//!
//! One can chain invariants, to get their desired mixin, like fallowing:
//!
//! 1. `NormalHttpUri<HttpUriSansFragment<HttpUri>>`: Invariant for http uris, which doesn't have fragment, and also in their normal form
//!
//!

use std::{borrow::Cow, convert::Infallible, ops::Deref};

use base::{
    _ext::uri::URIX,
    tdd::invariant::{InvariantRequirement, InvariantRequirementViolation, TypeInvariant},
};
use once_cell::sync::OnceCell;
use uriparse::{URIError, URI};

mod base;
pub mod component;
pub mod compound_invariant;
pub mod invariant;

/// A type representing valid http uri
#[derive(Debug, Clone)]
pub struct HttpUri<'uri> {
    uri: URI<'uri>,
    /// cache slot for stringified version
    str_cell: OnceCell<String>,
}

/// An enum of requirements for a uri to be valid http uri.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum HttpUriRequirement {
    #[strum(serialize = "Uri must have http(s) scheme")]
    UriMustHaveHttpScheme,
    #[strum(serialize = "Uri must have non empty host")]
    UriMustHaveNonEmptyHost,
}

impl InvariantRequirement for HttpUriRequirement {}

impl HttpUriRequirement {
    /// Creates error corresponding to requirement violation
    #[inline]
    pub fn err_violation<'uri>(&self, uri: URI<'uri>) -> HttpUriRequirementViolation<'uri> {
        InvariantRequirementViolation {
            value: uri,
            requirement: self.clone(),
        }
    }

    /// Asserts error is due to given requirement violation
    #[cfg(test)]
    #[inline]
    pub fn assert_violation<'a, 'uri>(&self, err: HttpUriRequirementViolation<'uri>) {
        assert_eq!(
            self, &err.requirement,
            "Expected violation of requirement: <{}>, but violated requirement: <{}>",
            self, &err.requirement
        )
    }
}

pub type HttpUriRequirementViolation<'uri> =
    InvariantRequirementViolation<URI<'uri>, HttpUriRequirement>;

#[allow(clippy::large_enum_variant)]
#[derive(Debug, thiserror::Error)]
pub enum InvalidHttpUriSource<'uri> {
    #[error("Http uri invariant requirement violation")]
    InvariantRequirementViolation(HttpUriRequirementViolation<'uri>),

    #[error("Given path str is invalid")]
    InvalidBaseSource(#[from] URIError),
}

impl<'uri> From<HttpUriRequirementViolation<'uri>> for InvalidHttpUriSource<'uri> {
    fn from(v: HttpUriRequirementViolation<'uri>) -> Self {
        Self::InvariantRequirementViolation(v)
    }
}

fn detect_any_violated_requirement(uri: &URI<'_>) -> Option<HttpUriRequirement> {
    // Ensure uri has http scheme
    if !HttpUri::HTTP_SCHEMES.contains(&uri.scheme().as_str()) {
        return Some(HttpUriRequirement::UriMustHaveHttpScheme);
    }
    // Ensure uri has valid non-empty host
    if !uri.has_non_empty_host() {
        return Some(HttpUriRequirement::UriMustHaveNonEmptyHost);
    }
    None
}

impl<'uri> TryFrom<URI<'uri>> for HttpUri<'uri> {
    type Error = HttpUriRequirementViolation<'uri>;

    fn try_from(uri: URI<'uri>) -> Result<Self, Self::Error> {
        match detect_any_violated_requirement(&uri) {
            Some(r) => Err(r.err_violation(uri)),
            None => Ok(Self {
                uri,
                str_cell: OnceCell::new(),
            }),
        }
    }
}

impl<'uri> TryFrom<&'uri str> for HttpUri<'uri> {
    type Error = InvalidHttpUriSource<'uri>;

    #[inline]
    fn try_from(uri_str: &'uri str) -> Result<Self, Self::Error> {
        let uri = URI::try_from(uri_str)?;
        Ok(Self::try_from(uri)?)
    }
}

/// Implementation in parity with remaining other invariants
impl<'a, 'uri, HUI> TryFrom<Cow<'a, HUI>> for HttpUri<'uri>
where
    HUI: TypeInvariant<HttpUri<'uri>> + Clone,
{
    type Error = Infallible;

    fn try_from(uri_invariant: Cow<'a, HUI>) -> Result<Self, Self::Error> {
        Ok(uri_invariant.into_owned().into())
    }
}

impl<'uri> Deref for HttpUri<'uri> {
    type Target = URI<'uri>;

    #[inline]
    fn deref(&self) -> &Self::Target {
        &self.uri
    }
}

impl<'uri> AsRef<str> for HttpUri<'uri> {
    #[inline]
    fn as_ref(&self) -> &str {
        self.str_cell.get_or_init(|| self.uri.to_string())
    }
}

#[allow(clippy::from_over_into)]
impl<'uri> Into<URI<'uri>> for HttpUri<'uri> {
    #[inline]
    fn into(self) -> URI<'uri> {
        self.uri
    }
}

impl<'uri> HttpUri<'uri> {
    const HTTP_SCHEME_DEFAULT_PORTS: [(&'static str, Option<u16>); 2] =
        [("http", Some(80)), ("https", Some(443))];

    const HTTP_SCHEMES: [&'static str; 2] = ["http", "https"];

    /// make it's lifetime static
    #[inline]
    pub fn into_owned(self) -> HttpUri<'static> {
        HttpUri {
            uri: self.uri.into_owned(),
            str_cell: self.str_cell.clone(),
        }
    }

    /// Normalizes inner uri per rfc
    #[inline]
    pub fn normalize_inner_uri(&mut self) {
        self.uri.normalize();
        self.str_cell = OnceCell::new();
    }

    /// Checks If explicit port of http uri is same as corresponding default port of it's http(s) scheme,
    #[inline]
    pub fn has_explicit_default_port(&self) -> bool {
        Self::HTTP_SCHEME_DEFAULT_PORTS.contains(&(self.scheme().as_str(), self.port()))
    }

    /// Removes explicit port, if explicit port of http uri is same as corresponding default port of it's http(s) scheme, Otherwise, this method is no op.
    #[inline]
    pub fn remove_explicit_default_port(&mut self) {
        if self.has_explicit_default_port() {
            self.uri.map_authority(|opt_authority| {
                opt_authority.map(|mut authority| {
                    // set port to None
                    authority.map_port(|_| None);
                    authority
                })
            });
            self.str_cell = OnceCell::new();
        }
    }

    /// Removes non-trailing empty segments.
    #[inline]
    pub fn remove_non_trailing_empty_segments(&mut self) {
        self.uri.remove_non_trailing_empty_segments();
        self.str_cell = OnceCell::new();
    }
}

/// Tests creation of [`HttpUri`].
#[cfg(test)]
pub mod tests_try_from {
    use claim::{assert_err, assert_matches, assert_ok};
    use rstest::rstest;

    use super::*;

    fn assert_requirement_violation(err: InvalidHttpUriSource, r: HttpUriRequirement) {
        match err {
            InvalidHttpUriSource::InvariantRequirementViolation(v) => r.assert_violation(v),
            InvalidHttpUriSource::InvalidBaseSource(_) => {
                panic!("Error is not due to requirement violation!")
            }
        }
    }

    #[rstest]
    #[case::invalid_char1("http://pod1.example.org/path/to/a b")]
    #[case::invalid_char2("http://pod1.example.org/a\nb")]
    #[case::invalid_char3("http://pod1.example.org/?a\\b")]
    #[case::invalid_char3("http://pod1.example.org/a%b")]
    #[case::invalid_char4("http://pod1.example.org/rama<>sita/doc")]
    #[case::invalid_non_ascii1("http://pod1.example.org/राम")]
    #[case::invalid_non_ascii2("http://pod1.example.org/అయోధ్య")]
    #[case::invalid_gen_delim3("http://pod1.example.org/a/b[c")]
    #[case::invalid_gen_delim4("http://pod1.example.org/a/b]c")]
    #[trace]
    fn invalid_uri_will_be_rejected(#[case] uri_str: &str) {
        assert_matches!(
            assert_err!(HttpUri::try_from(uri_str)),
            InvalidHttpUriSource::InvalidBaseSource(_)
        );
    }

    #[rstest]
    #[case::ftp("ftp://pod1.example.org/path/to/a")]
    #[case::urn("urn:pod1.example.org::ab")]
    #[case::data("data://pod1.example.org/?ab")]
    #[case::file("file://pod1.example.org/ab")]
    #[trace]
    fn uri_with_non_http_scheme_will_be_rejected(#[case] uri_str: &'static str) {
        assert_requirement_violation(
            assert_err!(HttpUri::try_from(uri_str)),
            HttpUriRequirement::UriMustHaveHttpScheme,
        );
    }

    #[rstest]
    #[case::http_no_authority("http:/path/to/a")]
    #[case::https_no_authority("https:/ab")]
    #[case::http_empty_host("http:///a/b?q")]
    #[case::https_empty_host("https:///a/b?q")]
    #[trace]
    fn uri_with_invalid_host_will_be_rejected(#[case] uri_str: &'static str) {
        assert_requirement_violation(
            assert_err!(HttpUri::try_from(uri_str)),
            HttpUriRequirement::UriMustHaveNonEmptyHost,
        );
    }

    #[rstest]
    #[case::origin_only("http://pod1.example.org")]
    #[case::explicit_root_path("http://pod1.example.org/")]
    #[case::un_normalized_scheme("HTTP://pod1.example.org/")]
    #[case::un_normalized_authority("http://pod1.EXAMPLE.org/")]
    #[case::ip_authority("http://127.0.0.1/")]
    #[case::localhost("http://localhost/")]
    #[case::explicit_default_port("http://pod1.example.org:80/")]
    #[case::https("https://pod1.example.org/")]
    #[case::explicit_default_port_2("https://pod1.example.org:443/")]
    #[case::with_query("http://pod1.example.org/a/b?q")]
    #[case::un_normalized_path_1("http://pod1.example.org/a//b")]
    #[case::un_normalized_path_2("http://pod1.example.org/a/b%41c")]
    #[case::un_normalized_path_3("http://pod1.example.org/c%2fd")]
    #[case::with_fragment("http://pod1.example.org/a#bc")]
    #[case::with_query_fragment("http://pod1.example.org/a?b#c")]
    #[case("http://pod1.example.org/a/b")]
    #[trace]
    fn valid_http_uri_will_be_accepted(#[case] uri_str: &'static str) {
        assert_ok!(HttpUri::try_from(uri_str));
    }
}

/// Tests algebra of [`HttpUri`].
#[cfg(test)]
pub mod tests_algebra {
    use rstest::rstest;

    use super::*;

    #[rstest]
    #[case::http("http://pod1.example.org/a/b?q", false)]
    #[case::http_80("http://pod1.example.org:80/a/b?q", true)]
    #[case::http_8000("http://pod1.example.org:8000/a/b?q", false)]
    #[case::https("https://pod1.example.org/a/b?q", false)]
    #[case::https_443("https://pod1.example.org:443/a/b?q", true)]
    #[case::https_743("https://pod1.example.org:743/a/b?q", false)]
    #[trace]
    fn test_explicit_default_port_check(
        #[case] source_uri_str: &'static str,
        #[case] expected: bool,
    ) {
        let http_uri = HttpUri::try_from(source_uri_str).unwrap();
        assert_eq!(http_uri.has_explicit_default_port(), expected);
    }

    #[rstest]
    #[case::http("http://pod1.example.org/a/b?q", "http://pod1.example.org/a/b?q")]
    #[case::http_80("http://pod1.example.org:80/a/b?q", "http://pod1.example.org/a/b?q")]
    #[case::http_8000(
        "http://pod1.example.org:8000/a/b?q",
        "http://pod1.example.org:8000/a/b?q"
    )]
    #[case::https("https://pod1.example.org/a/b?q", "https://pod1.example.org/a/b?q")]
    #[case::https_443("https://pod1.example.org:443/a/b?q", "https://pod1.example.org/a/b?q")]
    #[case::https_743(
        "https://pod1.example.org:743/a/b?q",
        "https://pod1.example.org:743/a/b?q"
    )]
    #[trace]
    fn test_explicit_default_port_removal(
        #[case] source_uri_str: &'static str,
        #[case] expected_uri_str: &'static str,
    ) {
        let mut http_uri = HttpUri::try_from(source_uri_str).unwrap();
        http_uri.remove_explicit_default_port();
        assert_eq!(http_uri.to_string(), expected_uri_str);
    }
}
