//! This module exports a generic invariant of http uri that has trailing slash in it's path

use crate::{define_generic_invariant, HttpUri};

/// An enum of requirements for a http uri to be with trailing slash.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum HttpUriWithTslashRequirement {
    #[strum(serialize = "Uri must have trailing slash in path")]
    UriMustHaveTrailingSlashInPath,
}

fn detect_any_violated_requirement(uri: &HttpUri<'_>) -> Option<HttpUriWithTslashRequirement> {
    // Ensure uri path has trailing slash
    if let Some(last_segment) = uri.path().segments().last() {
        // If path has a trailing slash, it implies it's last segment is empty
        if last_segment.is_empty() {
            return None;
        }
    }
    Some(HttpUriWithTslashRequirement::UriMustHaveTrailingSlashInPath)
}

define_generic_invariant!(
    /// A http uri, whose path has a trailing slash.
    HttpUriWithTslash,
    HttpUri,
    'uri,
    HttpUriWithTslashRequirement,
    HttpUriWithTslashRequirementViolation,
    detect_any_violated_requirement,
);

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_uri_with_tslash(
        uri_str: &'static str,
    ) -> Result<
        HttpUriWithTslash<'static, HttpUri<'static>>,
        HttpUriWithTslashRequirementViolation<'static, HttpUri<'static>>,
    > {
        let http_uri = HttpUri::try_from(uri_str).unwrap();
        Cow::<'static, HttpUri<'static>>::Owned(http_uri).try_into()
    }

    #[rstest]
    #[case("http://example.org/path/to/a")]
    #[case("https://example.org/path/to/b?c/")]
    #[case("https://example.org/path/to/b#")]
    #[trace]
    fn http_uri_with_out_tslash_will_be_rejected(#[case] uri_str: &'static str) {
        HttpUriWithTslashRequirement::UriMustHaveTrailingSlashInPath
            .assert_violation(assert_err!(try_uri_with_tslash(uri_str)))
    }

    #[rstest]
    #[case::implicit_root_slash1("http://example.org")]
    #[case::implicit_root_slash2("http://example.org#")]
    #[case::implicit_root_slash3("http://example.org?")]
    #[case("http://example.org/")]
    #[case("http://localhost/?a#f3")]
    #[case("https://example.org/path/to/a/?abc#f3")]
    #[trace]
    fn http_uri_with_tslash_will_be_accepted(#[case] uri_str: &'static str) {
        assert_ok!(try_uri_with_tslash(uri_str));
    }
}
