use std::borrow::Cow;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    invariant::normal::{NormalHttpUri, NormalHttpUriRequirement},
    HttpUri,
};

/// Concrete type for an http uri invariant, that is guaranteed to be normalized
pub type NormalizedHttpUri<'uri> = NormalHttpUri<'uri, HttpUri<'uri>>;

/// An enum of requirements for a http uri to be normalized.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum NormalizedHttpUriRequirement {
    #[strum(serialize = "HttpUri must be normal")]
    HttpUriMustBeNormal(NormalHttpUriRequirement),
}

impl From<NormalHttpUriRequirement> for NormalizedHttpUriRequirement {
    fn from(r: NormalHttpUriRequirement) -> Self {
        Self::HttpUriMustBeNormal(r)
    }
}

impl InvariantRequirement for NormalizedHttpUriRequirement {}

pub type NormalizedHttpUriRequirementViolation<'uri> =
    InvariantRequirementViolation<HttpUri<'uri>, NormalizedHttpUriRequirement>;

impl<'uri> TryFrom<HttpUri<'uri>> for NormalizedHttpUri<'uri> {
    type Error = NormalizedHttpUriRequirementViolation<'uri>;

    fn try_from(http_uri: HttpUri<'uri>) -> Result<Self, Self::Error> {
        NormalHttpUri::try_from(Cow::Owned(http_uri))
            .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
