use std::borrow::Cow;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    invariant::with_component::tslash::{HttpUriWithTslash, HttpUriWithTslashRequirement},
    HttpUri,
};

use super::hierarchical_normalized::{
    HierarchicalNormalizedHttpUri, HierarchicalNormalizedHttpUriRequirement,
};

/// Concrete type for an http uri invariant, that is guaranteed to be normalized, and is absolute (i.e. no fragment)
pub type HierarchicalNormalizedHttpUriWithTslash<'uri> =
    HttpUriWithTslash<'uri, HierarchicalNormalizedHttpUri<'uri>>;

/// An enum of requirements for a http uri to be absolute-normalized.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum HierarchicalNormalizedHttpUriWithTslashRequirement {
    #[strum(serialize = "HttpUri must be hierarchical and normalized")]
    HttpUriMustBeHierarchicalNormalized(HierarchicalNormalizedHttpUriRequirement),

    #[strum(serialize = "HttpUri must have path with trailing slash")]
    HttpUriMustBeWithTslash(HttpUriWithTslashRequirement),
}

impl From<HierarchicalNormalizedHttpUriRequirement>
    for HierarchicalNormalizedHttpUriWithTslashRequirement
{
    fn from(r: HierarchicalNormalizedHttpUriRequirement) -> Self {
        Self::HttpUriMustBeHierarchicalNormalized(r)
    }
}

impl From<HttpUriWithTslashRequirement> for HierarchicalNormalizedHttpUriWithTslashRequirement {
    fn from(r: HttpUriWithTslashRequirement) -> Self {
        Self::HttpUriMustBeWithTslash(r)
    }
}

impl InvariantRequirement for HierarchicalNormalizedHttpUriWithTslashRequirement {}

pub type HierarchicalNormalizedHttpUriWithTslashRequirementViolation<'uri> =
    InvariantRequirementViolation<
        HttpUri<'uri>,
        HierarchicalNormalizedHttpUriWithTslashRequirement,
    >;

impl<'uri> TryFrom<HttpUri<'uri>> for HierarchicalNormalizedHttpUriWithTslash<'uri> {
    type Error = HierarchicalNormalizedHttpUriWithTslashRequirementViolation<'uri>;

    fn try_from(http_uri: HttpUri<'uri>) -> Result<Self, Self::Error> {
        HttpUriWithTslash::try_from(Cow::Owned(
            HierarchicalNormalizedHttpUri::try_from(http_uri)
                .map_err(InvariantRequirementViolation::from_inner)?,
        ))
        .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
