use std::borrow::Cow;

use uriparse::Segment;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    component::segment::invariant::normal::{NormalSegment, NormalSegmentRequirement},
};

/// Concrete type for a segment invariant, that is guaranteed to be normalized
pub type NormalizedSegment<'segment> = NormalSegment<'segment, Segment<'segment>>;

/// An enum of requirements for a segment to be normalized.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum NormalizedSegmentRequirement {
    #[strum(serialize = "Segment must be normal")]
    SegmentMustBeNormal(NormalSegmentRequirement),
}

impl From<NormalSegmentRequirement> for NormalizedSegmentRequirement {
    fn from(r: NormalSegmentRequirement) -> Self {
        Self::SegmentMustBeNormal(r)
    }
}

impl InvariantRequirement for NormalizedSegmentRequirement {}

pub type NormalizedSegmentRequirementViolation<'segment> =
    InvariantRequirementViolation<Segment<'segment>, NormalizedSegmentRequirement>;

impl<'segment> TryFrom<Segment<'segment>> for NormalizedSegment<'segment> {
    type Error = NormalizedSegmentRequirementViolation<'segment>;

    fn try_from(segment: Segment<'segment>) -> Result<Self, Self::Error> {
        NormalSegment::try_from(Cow::Owned(segment))
            .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
