use std::borrow::Cow;

use uriparse::Segment;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    component::segment::invariant::non_dot::{NonDotSegment, NonDotSegmentRequirement},
};

use super::normalized::{NormalizedSegment, NormalizedSegmentRequirement};

/// Concrete type for a segment invariant, that is guaranteed to be normalized, and is non dot segment.
pub type CleanSegment<'segment> = NonDotSegment<'segment, NormalizedSegment<'segment>>;

/// An enum of requirements for a segment to be clean.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum CleanSegmentRequirement {
    #[strum(serialize = "Segment must be normalized")]
    SegmentMustBeNormalized(NormalizedSegmentRequirement),
    #[strum(serialize = "Segment must be non dot segment")]
    SegmentMustBeNonDotSegment(NonDotSegmentRequirement),
}

impl From<NormalizedSegmentRequirement> for CleanSegmentRequirement {
    fn from(r: NormalizedSegmentRequirement) -> Self {
        Self::SegmentMustBeNormalized(r)
    }
}

impl From<NonDotSegmentRequirement> for CleanSegmentRequirement {
    fn from(r: NonDotSegmentRequirement) -> Self {
        Self::SegmentMustBeNonDotSegment(r)
    }
}

impl InvariantRequirement for CleanSegmentRequirement {}

pub type CleanSegmentRequirementViolation<'segment> =
    InvariantRequirementViolation<Segment<'segment>, CleanSegmentRequirement>;

impl<'segment> TryFrom<Segment<'segment>> for CleanSegment<'segment> {
    type Error = CleanSegmentRequirementViolation<'segment>;

    fn try_from(segment: Segment<'segment>) -> Result<Self, Self::Error> {
        NonDotSegment::try_from(Cow::Owned(
            NormalizedSegment::try_from(segment)
                .map_err(InvariantRequirementViolation::from_inner)?,
        ))
        .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
