use std::borrow::Cow;

use uriparse::Path;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    component::path::invariant::rootless::{RootlessPath, RootlessPathRequirement},
};

use super::normalized::{NormalizedPath, NormalizedPathRequirement};

/// Concrete type for a path invariant, that is guaranteed to be normalized, and is rootless.
pub type RootlessNormalizedPath<'path> = RootlessPath<'path, NormalizedPath<'path>>;

/// An enum of requirements for a path to be rootless-normalized
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum RootlessNormalizedPathRequirement {
    #[strum(serialize = "Path must be normalized")]
    PathMustBeNormalized(NormalizedPathRequirement),
    #[strum(serialize = "Path must be rootless")]
    PathMustBeRootless(RootlessPathRequirement),
}

impl From<NormalizedPathRequirement> for RootlessNormalizedPathRequirement {
    fn from(r: NormalizedPathRequirement) -> Self {
        Self::PathMustBeNormalized(r)
    }
}

impl From<RootlessPathRequirement> for RootlessNormalizedPathRequirement {
    fn from(r: RootlessPathRequirement) -> Self {
        Self::PathMustBeRootless(r)
    }
}

impl InvariantRequirement for RootlessNormalizedPathRequirement {}

pub type RootlessNormalizedPathRequirementViolation<'path> =
    InvariantRequirementViolation<Path<'path>, RootlessNormalizedPathRequirement>;

impl<'path> TryFrom<Path<'path>> for RootlessNormalizedPath<'path> {
    type Error = RootlessNormalizedPathRequirementViolation<'path>;

    fn try_from(path: Path<'path>) -> Result<Self, Self::Error> {
        RootlessPath::try_from(Cow::Owned(
            NormalizedPath::try_from(path).map_err(InvariantRequirementViolation::from_inner)?,
        ))
        .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
