use std::borrow::Cow;

use uriparse::Path;

use crate::{
    base::tdd::invariant::{InvariantRequirement, InvariantRequirementViolation},
    component::path::invariant::absolute::{AbsolutePath, AbsolutePathRequirement},
};

use super::normalized::{NormalizedPath, NormalizedPathRequirement};

/// Concrete type for a path invariant, that is guaranteed to be normalized, and is absolute.
pub type AbsoluteNormalizedPath<'path> = AbsolutePath<'path, NormalizedPath<'path>>;

/// An enum of requirements for a path to be absolute-normalized
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum AbsoluteNormalizedPathRequirement {
    #[strum(serialize = "Path must be normalized")]
    PathMustBeNormalized(NormalizedPathRequirement),
    #[strum(serialize = "Path must be absolute")]
    PathMustBeAbsolute(AbsolutePathRequirement),
}

impl From<NormalizedPathRequirement> for AbsoluteNormalizedPathRequirement {
    fn from(r: NormalizedPathRequirement) -> Self {
        Self::PathMustBeNormalized(r)
    }
}

impl From<AbsolutePathRequirement> for AbsoluteNormalizedPathRequirement {
    fn from(r: AbsolutePathRequirement) -> Self {
        Self::PathMustBeAbsolute(r)
    }
}

impl InvariantRequirement for AbsoluteNormalizedPathRequirement {}

pub type AbsoluteNormalizedPathRequirementViolation<'path> =
    InvariantRequirementViolation<Path<'path>, AbsoluteNormalizedPathRequirement>;

impl<'path> TryFrom<Path<'path>> for AbsoluteNormalizedPath<'path> {
    type Error = AbsoluteNormalizedPathRequirementViolation<'path>;

    fn try_from(path: Path<'path>) -> Result<Self, Self::Error> {
        AbsolutePath::try_from(Cow::Owned(
            NormalizedPath::try_from(path).map_err(InvariantRequirementViolation::from_inner)?,
        ))
        .map_err(InvariantRequirementViolation::from_inner_cow)
    }
}
