use once_cell::sync::Lazy;
use regex::Regex;
use uriparse::Path;

/// A trait with helper methods on a uri path.
pub trait PathX {
    /// Tells if path has any non-trailing empty segment.
    fn has_non_trailing_empty_segments(&self) -> bool;

    /// Returns a new path with any non-trailing empty segments removed.
    // TODO should remove in place instead.
    fn remove_non_trailing_empty_segments(&self) -> Self;
}

static EMPTY_SEGMENT_RE: Lazy<Regex> = Lazy::new(|| Regex::new(r"//+").unwrap());

impl<'path> PathX for Path<'path> {
    fn has_non_trailing_empty_segments(&self) -> bool {
        let trailing_segment_index = self.segments().len() - 1;
        for (i, segment) in self.segments().iter().enumerate() {
            if segment.is_empty() && i != trailing_segment_index {
                return true;
            }
        }
        false
    }

    // TODO should remove in place instead.
    fn remove_non_trailing_empty_segments(&self) -> Self {
        let path_str = self.to_string();
        let path_str_normalized = &EMPTY_SEGMENT_RE.replace_all(&path_str, "/");
        Path::try_from(path_str_normalized.as_ref())
            .unwrap()
            .into_owned()
    }
}

#[cfg(test)]
pub mod tests {
    use rstest::rstest;

    use super::*;

    #[rstest]
    #[case::empty_path_str("", false)]
    #[case::root("/", false)]
    #[case::leading_slash("/a/b/c", false)]
    #[case::trailing_slash("a/b/c/", false)]
    #[case::lt_slash("/ab/c/", false)]
    #[case::leading_double_slash("//", true)]
    #[case::even_slash("a//b/c", true)]
    #[case::odd_slash("a/b///c/d", true)]
    #[trace]
    fn test_non_trailing_empty_segments_check(#[case] path_str: &str, #[case] expected: bool) {
        let path = Path::try_from(path_str).unwrap();
        assert_eq!(path.has_non_trailing_empty_segments(), expected);
    }

    #[rstest]
    #[case::empty_path_str("", "")]
    #[case::root("/", "/")]
    #[case::leading_slash("/a/b/c", "/a/b/c")]
    #[case::trailing_slash("a/b/c/", "a/b/c/")]
    #[case::lt_slash("/ab/c/", "/ab/c/")]
    #[case::leading_double_slash("//", "/")]
    #[case::even_slash("a//b/c", "a/b/c")]
    #[case::odd_slash("a/b///c/d", "a/b/c/d")]
    #[trace]
    fn test_non_trailing_empty_segments_removal(
        #[case] path_str: &str,
        #[case] expected_path_str: &str,
    ) {
        let path = Path::try_from(path_str).unwrap();
        let normal_path = path.remove_non_trailing_empty_segments();
        assert_eq!(expected_path_str, normal_path.to_string());
    }
}
