//! This module exports a generic invariant of segment that is normal

use std::{borrow::Cow, marker::PhantomData};

use percent_encoding::utf8_percent_encode;
use uriparse::Segment;

use crate::{component::character::pchar::PCHAR_PCT_ENCODE_SET, define_generic_invariant};

/// An enum of requirements for a segment to be non empty.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum NormalSegmentRequirement {
    #[strum(serialize = "Segment must be normal per rfc")]
    SegmentMustBeNormalPerRfc,
}

fn detect_any_violated_requirement(segment: &Segment<'_>) -> Option<NormalSegmentRequirement> {
    // Ensure segment is normal
    if !segment.is_normalized() {
        return Some(NormalSegmentRequirement::SegmentMustBeNormalPerRfc);
    }
    None
}

define_generic_invariant!(
    /// A segment that is normal as per rfc
    NormalSegment,
    Segment,
    'segment,
    NormalSegmentRequirement,
    NormalSegmentRequirementViolation,
    detect_any_violated_requirement,
);

impl<'segment> NormalSegment<'segment, Segment<'segment>> {
    #[inline]
    pub fn normalized_from(mut segment: Segment<'segment>) -> Self {
        segment.normalize();
        NormalSegment {
            inner_invariant: segment,
            _phantom: PhantomData,
        }
    }

    pub fn normalized_from_unencoded_str(segment_str: &'segment str) -> Self {
        let resolved_segment = if let Cow::Owned(encoded_segment_str) =
            utf8_percent_encode(segment_str, PCHAR_PCT_ENCODE_SET).into()
        {
            let segment = Segment::try_from(encoded_segment_str.as_str()).unwrap();
            segment.into_owned()
        } else {
            Segment::try_from(segment_str).unwrap()
        };
        Self::normalized_from(resolved_segment)
    }
}

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_normal_segment(
        segment_str: &'static str,
    ) -> Result<
        NormalSegment<'static, Segment<'static>>,
        NormalSegmentRequirementViolation<'static, Segment<'static>>,
    > {
        let segment = Segment::try_from(segment_str).unwrap();
        Cow::<'static, Segment<'static>>::Owned(segment).try_into()
    }

    #[rstest]
    #[case::a("%61yodhya")]
    #[case::a_cap("%41yodhya")]
    #[case::dig_1("raghu%31")]
    #[case::hyphen("rama%2Drajya")]
    #[case::tilde("rama%7Esita")]
    #[case::period("kosala%2Eayodhya")]
    #[case::period_2("%2E%2E")]
    #[case::underscore("rama%5Flakshmana")]
    #[trace]
    fn un_reserved_char_encoded_segment_will_be_rejected(#[case] segment_str: &'static str) {
        NormalSegmentRequirement::SegmentMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_segment(segment_str)));
    }

    #[rstest]
    #[case("rama%3ddharma")]
    #[case("rama%2blakshmana")]
    #[case("rama%2clakshmana")]
    #[case("%e0%A4%b0%E0%A4%BE%E0%A4%AE")]
    #[case("kosala%2fayodhya")]
    #[trace]
    fn lowercase_pct_encoded_segment_will_be_rejected(#[case] segment_str: &'static str) {
        NormalSegmentRequirement::SegmentMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_segment(segment_str)));
    }

    #[rstest]
    #[case::empty("")]
    #[case::dot(".")]
    #[case::dot2("..")]
    #[case("ayodhya")]
    #[case::un_reserved_unencoded_1("a.acl")]
    #[case::un_reserved_unencoded_2("a~acl")]
    #[case::un_reserved_unencoded_3("a_acl")]
    #[case::un_reserved_unencoded_4("a-acl")]
    #[case::sub_delim_unencoded_1("a$acl")]
    #[case::sub_delim_unencoded_2("rama=dharma")]
    #[case::sub_delim_unencoded_3("rama+lakshmana")]
    #[case::sub_delim_unencoded_4("rama,lakshmana")]
    #[case::sub_delim_unencoded_5("rama&lakshmana")]
    #[case::sub_delim_encoded_1("%24acl")]
    #[case::sub_delim_encoded_2("rama%3Ddharma")]
    #[case::sub_delim_encoded_3("rama%2Blakshmana")]
    #[case::sub_delim_encoded_4("rama%2Clakshmana")]
    #[case::sub_delim_encoded_5("rama%26lakshmana")]
    #[case::excepted_gen_delim_unencoded_1("a:b")]
    #[case::expected_gen_delim_unencoded_2("a@b")]
    #[case::excepted_gen_delim_encoded_1("a%3Ab")]
    #[case::expected_gen_delim_encoded_2("a%40b")]
    #[case::gen_delim_encoded_1("b%2Fc")]
    #[case::gen_delim_encoded_2("b%3Fc")]
    #[case::gen_delim_encoded_3("b%23c")]
    #[case::gen_delim_encoded_4("b%5B%5Dc")]
    #[case::non_ascii_pct_encoded("%E0%A4%B0%E0%A4%BE%E0%A4%AE")]
    #[case::non_ascii_pct_encoded("%E0%B0%85%E0%B0%AF%E0%B1%8B%E0%B0%A7%E0%B1%8D%E0%B0%AF")]
    #[trace]
    fn normalized_segment_will_be_accepted(#[case] segment_str: &'static str) {
        assert_ok!(try_normal_segment(segment_str));
    }
}

/// Tests converting given valid segment to `NormalSegment`, while normalizing in the way.
#[cfg(test)]
pub mod tests_normalized_from {
    use rstest::rstest;

    use super::*;

    fn assert_correct_normalization(
        source_segment_str: &'static str,
        expected_segment_str: &'static str,
    ) {
        let source_segment: Segment<'static> = Segment::try_from(source_segment_str).unwrap();
        let segment = NormalSegment::normalized_from(source_segment);
        assert_eq!(segment.as_str(), expected_segment_str);
    }

    #[rstest]
    #[case::a("%61yodhya", "ayodhya")]
    #[case::a_cap("%41yodhya", "Ayodhya")]
    #[case::dig_1("raghu%31", "raghu1")]
    #[case::hyphen("rama%2Drajya", "rama-rajya")]
    #[case::tilde("rama%7Esita", "rama~sita")]
    #[case::period("kosala%2Eayodhya", "kosala.ayodhya")]
    #[case::period_2("%2E%2E", "..")]
    #[case::underscore("rama%5Flakshmana", "rama_lakshmana")]
    #[trace]
    fn unreserved_chars_will_be_normalized_correctly(
        #[case] source_segment_str: &'static str,
        #[case] expected_segment_str: &'static str,
    ) {
        assert_correct_normalization(source_segment_str, expected_segment_str);
    }

    #[rstest]
    #[case("rama%3ddharma", "rama%3Ddharma")]
    #[case("rama%2blakshmana", "rama%2Blakshmana")]
    #[case("rama%2clakshmana", "rama%2Clakshmana")]
    #[case("%e0%A4%b0%E0%A4%BE%E0%A4%AE", "%E0%A4%B0%E0%A4%BE%E0%A4%AE")]
    #[case("kosala%2fayodhya", "kosala%2Fayodhya")]
    #[trace]
    fn pct_encoded_octet_case_will_be_normalized_correctly(
        #[case] source_segment_str: &'static str,
        #[case] expected_segment_str: &'static str,
    ) {
        assert_correct_normalization(source_segment_str, expected_segment_str);
    }
}

/// Tests converting given unencoded string to `NormalSegment`, while pct-encoding, and normalizing in the way.
#[cfg(test)]
pub mod tests_normalized_from_unencoded_str {
    use rstest::rstest;

    use super::*;

    fn assert_correct_normalization(
        source_segment_str: &'static str,
        expected_segment_str: &'static str,
    ) {
        let segment = NormalSegment::normalized_from_unencoded_str(source_segment_str);
        assert_eq!(segment.as_str(), expected_segment_str);
    }

    #[rstest]
    #[case::invalid_char1("a b", "a%20b")]
    #[case::invalid_char2("a\nb", "a%0Ab")]
    #[case::invalid_char3("a\\b", "a%5Cb")]
    #[case::invalid_char3("a%b", "a%25b")]
    #[case::invalid_char4("rama<>sita/doc", "rama%3C%3Esita%2Fdoc")]
    #[case::invalid_non_ascii1("राम", "%E0%A4%B0%E0%A4%BE%E0%A4%AE")]
    #[case::invalid_non_ascii2("అయోధ్య", "%E0%B0%85%E0%B0%AF%E0%B1%8B%E0%B0%A7%E0%B1%8D%E0%B0%AF")]
    #[case::invalid_gen_delim3("a/b[c", "a%2Fb%5Bc")]
    #[case::invalid_gen_delim4("a/b]c", "a%2Fb%5Dc")]
    #[case::uri(
        "http://pod1.example.org/path/to/a",
        "http:%2F%2Fpod1.example.org%2Fpath%2Fto%2Fa"
    )]
    #[trace]
    fn segment_str_will_be_properly_pct_encoded_and_normalized(
        #[case] source_segment_str: &'static str,
        #[case] expected_segment_str: &'static str,
    ) {
        assert_correct_normalization(source_segment_str, expected_segment_str);
    }
}
