//! This module exports a generic invariant of segment that is non dot segment

use uriparse::Segment;

use crate::define_generic_invariant;

/// An enum of requirements for a segment to be non dot segment.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum NonDotSegmentRequirement {
    #[strum(serialize = "Segment must not be dot segment")]
    SegmentMustNotBeDotSegment,
}

fn detect_any_violated_requirement(segment: &Segment<'_>) -> Option<NonDotSegmentRequirement> {
    // Ensure segment is non dot segment
    if segment.is_dot_segment() {
        return Some(NonDotSegmentRequirement::SegmentMustNotBeDotSegment);
    }
    None
}

define_generic_invariant!(
    /// A segment that is not a dot segment.
    NonDotSegment,
    Segment,
    'segment,
    NonDotSegmentRequirement,
    NonDotSegmentRequirementViolation,
    detect_any_violated_requirement,
);

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_non_dot_segment(
        segment_str: &'static str,
    ) -> Result<
        NonDotSegment<'static, Segment<'static>>,
        NonDotSegmentRequirementViolation<'static, Segment<'static>>,
    > {
        let segment = Segment::try_from(segment_str).unwrap();
        Cow::<'static, Segment<'static>>::Owned(segment).try_into()
    }

    #[rstest]
    #[case(".")]
    #[case("..")]
    #[case("%2e")]
    #[case("%2e%2e")]
    #[trace]
    fn dot_segment_will_be_rejected(#[case] segment_str: &'static str) {
        NonDotSegmentRequirement::SegmentMustNotBeDotSegment
            .assert_violation(assert_err!(try_non_dot_segment(segment_str)))
    }

    #[rstest]
    #[case("abc")]
    #[case("...")]
    #[case("image.png")]
    #[trace]
    fn non_dot_segment_will_be_accepted(#[case] segment_str: &'static str) {
        assert_ok!(try_non_dot_segment(segment_str));
    }
}
