//! This module exports a generic invariant of normal path

use uriparse::Path;

use crate::{base::_ext::path::PathX, define_generic_invariant};

/// An enum of requirements for a path to be normal.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum NormalPathRequirement {
    #[strum(serialize = "Path must be normal per rfc")]
    PathMustBeNormalPerRfc,
    #[strum(serialize = "Path must not have non trailing empty segment")]
    PathMustNotHaveNonTrailingEmptySegment,
}

fn detect_any_violated_requirement(path: &Path<'_>) -> Option<NormalPathRequirement> {
    // Ensure path is normal
    if !path.is_normalized(false) {
        return Some(NormalPathRequirement::PathMustBeNormalPerRfc);
    }
    // Ensure path doesn't has non trailing empty segments
    if path.has_non_trailing_empty_segments() {
        return Some(NormalPathRequirement::PathMustNotHaveNonTrailingEmptySegment);
    }
    None
}

define_generic_invariant!(
    /// An invariant of path, that is normal as per rfc3986
    NormalPath,
    Path,
    'path,
    NormalPathRequirement,
    NormalPathRequirementViolation,
    detect_any_violated_requirement,
);

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_normal_path(
        path_str: &'static str,
    ) -> Result<
        NormalPath<'static, Path<'static>>,
        NormalPathRequirementViolation<'static, Path<'static>>,
    > {
        let path = Path::try_from(path_str).unwrap();
        Cow::<'static, Path<'static>>::Owned(path).try_into()
    }

    #[rstest]
    #[case::a("kosala/%61yodhya")]
    #[case::a_cap("/kosala/%41yodhya")]
    #[case::dig_1("kosala/raghu%31")]
    #[case::hyphen("/rama%2Drajya")]
    #[case::tilde("rama%7Esita")]
    #[case::period("/kosala%2Eayodhya")]
    #[case::period_2("path/to/%2E%2E/c")]
    #[case::underscore("/rama%5Flakshmana")]
    #[trace]
    fn un_reserved_char_encoded_path_will_be_rejected(#[case] path_str: &'static str) {
        NormalPathRequirement::PathMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_path(path_str)));
    }

    #[rstest]
    #[case("rama%3ddharma")]
    #[case("/rama%2blakshmana")]
    #[case("rama%2clakshmana")]
    #[case("/%e0%A4%b0%E0%A4%BE%E0%A4%AE")]
    #[case("kosala%2fayodhya")]
    #[trace]
    fn lowercase_pct_encoded_path_will_be_rejected(#[case] path_str: &'static str) {
        NormalPathRequirement::PathMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_path(path_str)));
    }

    #[rstest]
    #[case(".")]
    #[case("..")]
    #[case("/.")]
    #[case("/..")]
    #[case("./path/to/a")]
    #[case("../a")]
    #[case("path/to/a/../b/")]
    #[case("/path/to/c/.")]
    #[case("path/to/c/./")]
    #[case("/path/to/c/..")]
    #[case("path/to/c/../")]
    #[trace]
    fn path_with_dot_segments_will_be_rejected(#[case] path_str: &'static str) {
        NormalPathRequirement::PathMustBeNormalPerRfc
            .assert_violation(assert_err!(try_normal_path(path_str)));
    }

    #[rstest]
    #[case("path/to//a")]
    #[case("path/to//")]
    #[case("/path/to//a")]
    #[case("/path/to//")]
    #[trace]
    fn path_with_non_trailing_empty_segment_will_be_rejected(#[case] path_str: &'static str) {
        NormalPathRequirement::PathMustNotHaveNonTrailingEmptySegment
            .assert_violation(assert_err!(try_normal_path(path_str)));
    }

    #[rstest]
    #[case::empty("")]
    #[case("kosala/ayodhya")]
    #[case("path/to/container/")]
    #[case::un_reserved_unencoded_1("path/to/container/.acl")]
    #[case::un_reserved_unencoded_2("/path/to/container/~acl")]
    #[case::un_reserved_unencoded_3("path/to/container/_acl")]
    #[case::un_reserved_unencoded_4("/path/to/container/-acl")]
    #[case::sub_delim_unencoded_1("path/to/container/$acl")]
    #[case::sub_delim_unencoded_2("/rama=dharma")]
    #[case::sub_delim_unencoded_3("rama+lakshmana")]
    #[case::sub_delim_unencoded_4("/rama,lakshmana")]
    #[case::sub_delim_unencoded_5("rama&lakshmana")]
    #[case::sub_delim_encoded_1("/path/to/container/%24acl")]
    #[case::sub_delim_encoded_2("rama%3Ddharma")]
    #[case::sub_delim_encoded_3("/rama%2Blakshmana")]
    #[case::sub_delim_encoded_4("rama%2Clakshmana")]
    #[case::sub_delim_encoded_5("/rama%26lakshmana")]
    #[case::excepted_gen_delim_unencoded_1("a:b")]
    #[case::expected_gen_delim_unencoded_2("a@b")]
    #[case::excepted_gen_delim_encoded_1("/a%3Ab")]
    #[case::expected_gen_delim_encoded_2("a%40b")]
    #[case::gen_delim_encoded_1("/a/b%2Fc")]
    #[case::gen_delim_encoded_2("a/b%3Fc")]
    #[case::gen_delim_encoded_3("/a/b%23c")]
    #[case::gen_delim_encoded_4("a/b%5B%5Dc")]
    #[case::non_ascii_pct_encoded("/ramayana/%E0%A4%B0%E0%A4%BE%E0%A4%AE")]
    #[case::non_ascii_pct_encoded("%E0%B0%85%E0%B0%AF%E0%B1%8B%E0%B0%A7%E0%B1%8D%E0%B0%AF")]
    #[trace]
    fn normalized_path_will_be_accepted(#[case] path_str: &'static str) {
        assert_ok!(try_normal_path(path_str));
    }
}
