//! This module exports a generic invariant of absolute path

use uriparse::Path;

use crate::define_generic_invariant;

/// An enum of requirements for a path to be absolute.
#[derive(Debug, strum_macros::Display, Clone, PartialEq)]
pub enum AbsolutePathRequirement {
    #[strum(serialize = "Path must be absolute")]
    PathMustBeAbsolute,
}

fn detect_any_violated_requirement(path: &Path<'_>) -> Option<AbsolutePathRequirement> {
    // Ensure path is absolute
    if !path.is_absolute() {
        return Some(AbsolutePathRequirement::PathMustBeAbsolute);
    }
    None
}

define_generic_invariant!(
    /// An invariant of path, that is absolute as per rfc3986
    AbsolutePath,
    Path,
    'path,
    AbsolutePathRequirement,
    AbsolutePathRequirementViolation,
    detect_any_violated_requirement,
);

#[cfg(test)]
pub mod test_try_from {
    use std::borrow::Cow;

    use claim::{assert_err, assert_ok};
    use rstest::rstest;

    use super::*;

    fn try_absolute_path(
        path_str: &'static str,
    ) -> Result<
        AbsolutePath<'static, Path<'static>>,
        AbsolutePathRequirementViolation<'static, Path<'static>>,
    > {
        let path = Path::try_from(path_str).unwrap();
        Cow::<'static, Path<'static>>::Owned(path).try_into()
    }

    #[rstest]
    #[case("")]
    #[case(".")]
    #[case("abc")]
    #[case("abc/def")]
    #[trace]
    fn non_absolute_path_will_be_rejected(#[case] path_str: &'static str) {
        AbsolutePathRequirement::PathMustBeAbsolute
            .assert_violation(assert_err!(try_absolute_path(path_str)))
    }

    #[rstest]
    #[case("/")]
    #[case("/..")]
    #[case("/abc")]
    #[case("/abc/def/")]
    #[trace]
    fn absolute_path_will_be_accepted(#[case] path_str: &'static str) {
        assert_ok!(try_absolute_path(path_str));
    }
}
