#[macro_export(local_inner_macros)]
macro_rules! define_generic_invariant {
    (
        $(#[$outer:meta])*
        $Invariant:ident,
        $BaseType:ident,
        $base_type_lifetime_label:lifetime,
        $InvariantRequirement:ident,
        $InvariantRequirementViolation:ident,
        $violation_detector:ident,
    ) => {
        pub use ws::$Invariant;
        pub use ws::$InvariantRequirementViolation;

        mod ws {
            use std::{
                borrow::{Borrow, Cow},
                marker::PhantomData,
                ops::Deref,
            };

            use $crate::{
                base::tdd::invariant::{
                    InvariantRequirement, InvariantRequirementViolation, TypeInvariant,
                },
            };

            use super::{$InvariantRequirement, $violation_detector, $BaseType};

            impl InvariantRequirement for $InvariantRequirement {}

            impl $InvariantRequirement {
                /// Creates error corresponding to requirement violation
                #[inline]
                pub fn err_violation<'a, TI>(
                    &self,
                    value: Cow<'a, TI>,
                ) -> $InvariantRequirementViolation<'a, TI>
                where
                    TI: Clone,
                {
                    InvariantRequirementViolation {
                        value,
                        requirement: self.clone(),
                    }
                }

                /// Asserts error is due to given requirement violation
                #[cfg(test)]
                #[inline]
                pub fn assert_violation<'a, TI>(
                    &self,
                    err: $InvariantRequirementViolation<'a, TI>,
                ) where
                    TI: Clone,
                {
                    std::assert_eq!(
                        self, &err.requirement,
                        "Expected violation of requirement: <{}>, but violated requirement: <{}>",
                        self, &err.requirement
                    );
                }
            }

            pub type $InvariantRequirementViolation<'a, TI> =
                InvariantRequirementViolation<Cow<'a, TI>, $InvariantRequirement>;

            $(#[$outer])*
            #[derive(Debug, Clone)]
            pub struct $Invariant<$base_type_lifetime_label, TI>
            where
                TI: TypeInvariant<$BaseType<$base_type_lifetime_label>> + Clone,
            {
                pub(super) inner_invariant: TI,
                pub(super) _phantom: PhantomData<$BaseType<$base_type_lifetime_label>>,
            }

            impl<'a, $base_type_lifetime_label, TI> TryFrom<Cow<'a, TI>> for $Invariant<$base_type_lifetime_label, TI>
            where
                TI: TypeInvariant<$BaseType<$base_type_lifetime_label>> + Clone,
            {
                type Error = $InvariantRequirementViolation<'a, TI>;

                fn try_from(inner_invariant: Cow<'a, TI>) -> Result<Self, Self::Error> {
                    let uri_ref: &$BaseType<$base_type_lifetime_label> = inner_invariant.as_ref().borrow();

                    match $violation_detector(uri_ref) {
                        Some(r) => Err(r.err_violation(inner_invariant)),

                        None => Ok(Self {
                            inner_invariant: inner_invariant.into_owned(),
                            _phantom: PhantomData,
                        }),
                    }
                }
            }

            impl<$base_type_lifetime_label, TI> Deref for $Invariant<$base_type_lifetime_label, TI>
            where
                TI: TypeInvariant<$BaseType<$base_type_lifetime_label>> + Clone,
            {
                type Target = $BaseType<$base_type_lifetime_label>;

                fn deref(&self) -> &Self::Target {
                    self.inner_invariant.borrow()
                }
            }

            impl<$base_type_lifetime_label, TI> Borrow<$BaseType<$base_type_lifetime_label>> for $Invariant<$base_type_lifetime_label, TI>
            where
                TI: TypeInvariant<$BaseType<$base_type_lifetime_label>> + Clone,
            {
                fn borrow(&self) -> &$BaseType<$base_type_lifetime_label> {
                    self.inner_invariant.borrow()
                }
            }

            #[allow(clippy::from_over_into)]
            impl<$base_type_lifetime_label, TI> Into<$BaseType<$base_type_lifetime_label>> for $Invariant<$base_type_lifetime_label, TI>
            where
                TI: TypeInvariant<$BaseType<$base_type_lifetime_label>> + Clone,
            {
                fn into(self) -> $BaseType<$base_type_lifetime_label> {
                    self.inner_invariant.into()
                }
            }

            impl<$base_type_lifetime_label, TI> $Invariant<$base_type_lifetime_label, TI>
            where
                TI: TypeInvariant<$BaseType<$base_type_lifetime_label>> + Clone,
            {
                #[inline]
                pub fn into_inner(self) -> TI {
                    self.inner_invariant
                }
            }

            impl<$base_type_lifetime_label, TI> TypeInvariant<$BaseType<$base_type_lifetime_label>> for $Invariant<$base_type_lifetime_label, TI> where
                TI: TypeInvariant<$BaseType<$base_type_lifetime_label>> + Clone
            {
            }
        }
    };
}
