use uriparse::{Host, URI};

use super::path::PathX;

/// A trait with helper methods on a uri.
#[allow(clippy::upper_case_acronyms)]
pub trait URIX {
    /// Checks if uri has non empty host
    fn has_non_empty_host(&self) -> bool;

    /// Checks if uri's path has non-trailing empty segments.
    fn has_non_trailing_empty_segments(&self) -> bool;

    /// Removes non-trailing empty segments from uri's path.
    /// If no such, this method is no op.
    fn remove_non_trailing_empty_segments(&mut self);
}

impl<'uri> URIX for URI<'uri> {
    fn has_non_empty_host(&self) -> bool {
        match self.host() {
            Some(host) => match host {
                // Currently empty host is of `RegisteredName` invariant in [`uriparse`] crate
                Host::RegisteredName(host_reg) => !host_reg.as_str().is_empty(),
                _ => true,
            },
            None => false,
        }
    }

    /// Checks if uri's path has non-trailing empty segments.
    #[inline]
    fn has_non_trailing_empty_segments(&self) -> bool {
        self.path().has_non_trailing_empty_segments()
    }

    /// Removes non-trailing empty segments from uri's path.
    /// If no such, this method is no op.
    #[inline]
    fn remove_non_trailing_empty_segments(&mut self) {
        if self.has_non_trailing_empty_segments() {
            self.set_path(self.path().remove_non_trailing_empty_segments())
                .unwrap();
        }
    }
}

#[cfg(test)]
pub mod tests {
    use rstest::rstest;
    use uriparse::URI;

    use super::*;

    #[rstest]
    #[case::http("http://pod1.example.org/a/b?q", true)]
    #[case::https("https://pod1.example.org/a/b?q", true)]
    #[case::file("file:///a/b", false)]
    #[case::ftp("ftp://pod1.example.org/a/b?q", true)]
    #[case::http_no_authority("http:/path/to/a", false)]
    #[case::https_no_authority("https:/ab", false)]
    #[case::http_empty_host("http:///a/b?q", false)]
    #[case::https_empty_host("https:///a/b?q", false)]
    #[trace]
    fn test_non_empty_host_check(#[case] source_uri_str: &'static str, #[case] expected: bool) {
        let uri = URI::try_from(source_uri_str).unwrap();
        assert_eq!(uri.has_non_empty_host(), expected);
    }

    #[rstest]
    #[case::empty_path_str("http://pod1.example.org", "http://pod1.example.org/")]
    #[case::root("http://pod1.example.org/", "http://pod1.example.org/")]
    #[case("http://pod1.example.org/a/b/c", "http://pod1.example.org/a/b/c")]
    #[case::lt_slash("http://pod1.example.org/ab/c/", "http://pod1.example.org/ab/c/")]
    #[case::leading_double_slash("http://pod1.example.org//", "http://pod1.example.org/")]
    #[case::even_slash("http://pod1.example.org/a//b/c", "http://pod1.example.org/a/b/c")]
    #[case::odd_slash("http://pod1.example.org/a/b///c/d", "http://pod1.example.org/a/b/c/d")]
    #[trace]
    fn test_non_trailing_empty_segments_removal(
        #[case] source_uri_str: &'static str,
        #[case] expected_uri_str: &'static str,
    ) {
        let mut uri = URI::try_from(source_uri_str).unwrap();
        uri.remove_non_trailing_empty_segments();
        assert_eq!(uri.to_string(), expected_uri_str);
    }
}
