use http::Uri;

use crate::RequestUri;

impl<'a> TryFrom<&'a Uri> for RequestUri<'a> {
    type Error = &'static str;

    fn try_from(uri: &'a Uri) -> Result<Self, Self::Error> {
        if let Some(scheme) = uri.scheme() {
            if let Some(authority) = uri.authority() {
                let (mut username, mut password) = (None, None);

                let mut split = authority.as_str().split('@').rev();
                if split.next().is_none() {
                    debug_assert!(false, "unreachable");
                    return Err("authority invalid");
                }

                if let Some(username_password) = split.next() {
                    if split.next().is_some() {
                        debug_assert!(false, "unreachable");
                        return Err("authority invalid");
                    }

                    let mut split = username_password.split(':');
                    let username_tmp = split.next().ok_or("username missing")?;
                    if !username_tmp.is_empty() {
                        username = Some(username_tmp);
                    }
                    if let Some(password_tmp) = split.next() {
                        if split.next().is_some() {
                            debug_assert!(false, "unreachable");
                            return Err("authority invalid");
                        }

                        if !password_tmp.is_empty() {
                            password = Some(password_tmp);
                        }
                    }
                }

                Ok(Self::AbsoluteUri {
                    scheme: scheme.as_str(),
                    username,
                    password,
                    host: authority.host(),
                    port: authority.port_u16(),
                    path: uri.path(),
                    query: uri.query(),
                    fragment: None,
                })
            } else {
                Err("authority missing")
            }
        } else if let Some(authority) = uri.authority() {
            Ok(Self::Authority {
                host: authority.host(),
                port: authority.port_u16(),
            })
        } else if uri.path() == "*" {
            Ok(Self::Asterisk)
        } else {
            if !uri.path().starts_with('/') {
                debug_assert!(false, "unreachable");
                return Err("path invalid");
            }

            Ok(Self::AbsPath {
                path: uri.path(),
                query: uri.query(),
                fragment: None,
            })
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_from_uri() {
        //
        assert_eq!(
            RequestUri::try_from(&"*".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::Asterisk
        );

        //
        assert_eq!(
            RequestUri::try_from(
                &"http://www.w3.org/pub/WWW/TheProject.html"
                    .parse::<Uri>()
                    .unwrap()
            )
            .unwrap(),
            RequestUri::AbsoluteUri {
                scheme: "http",
                username: None,
                password: None,
                host: "www.w3.org",
                port: None,
                path: "/pub/WWW/TheProject.html",
                query: None,
                fragment: None
            }
        );
        assert_eq!(
            RequestUri::try_from(&"http://www.w3.org".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::AbsoluteUri {
                scheme: "http",
                username: None,
                password: None,
                host: "www.w3.org",
                port: None,
                path: "/",
                query: None,
                fragment: None
            }
        );
        assert_eq!(
            RequestUri::try_from(
                &"http://username:password@www.w3.org"
                    .parse::<Uri>()
                    .unwrap()
            )
            .unwrap(),
            RequestUri::AbsoluteUri {
                scheme: "http",
                username: Some("username"),
                password: Some("password"),
                host: "www.w3.org",
                port: None,
                path: "/",
                query: None,
                fragment: None
            }
        );
        assert_eq!(
            RequestUri::try_from(&"http://username@www.w3.org".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::AbsoluteUri {
                scheme: "http",
                username: Some("username"),
                password: None,
                host: "www.w3.org",
                port: None,
                path: "/",
                query: None,
                fragment: None
            }
        );
        assert_eq!(
            RequestUri::try_from(&"http://:password@www.w3.org".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::AbsoluteUri {
                scheme: "http",
                username: None,
                password: Some("password"),
                host: "www.w3.org",
                port: None,
                path: "/",
                query: None,
                fragment: None
            }
        );

        //
        assert_eq!(
            RequestUri::try_from(&"/pub/WWW/TheProject.html".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::AbsPath {
                path: "/pub/WWW/TheProject.html",
                query: None,
                fragment: None
            }
        );
        assert_eq!(
            RequestUri::try_from(&"/".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::AbsPath {
                path: "/",
                query: None,
                fragment: None
            }
        );
        assert_eq!(
            RequestUri::try_from(&"/foo?bar=".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::AbsPath {
                path: "/foo",
                query: Some("bar="),
                fragment: None
            }
        );

        //
        assert_eq!(
            RequestUri::try_from(&"proxy.com".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::Authority {
                host: "proxy.com",
                port: None
            }
        );
        assert_eq!(
            RequestUri::try_from(&"proxy.com:443".parse::<Uri>().unwrap()).unwrap(),
            RequestUri::Authority {
                host: "proxy.com",
                port: Some(443)
            }
        );
    }
}
