//! https://datatracker.ietf.org/doc/html/rfc2616#section-5.1.2

use http::Uri;

#[derive(Debug, Copy, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum RequestUri {
    Asterisk,
    AbsoluteUri,
    AbsPath,
    Authority,
}

impl From<&Uri> for RequestUri {
    fn from(uri: &Uri) -> Self {
        if uri.scheme().is_some() {
            Self::AbsoluteUri
        } else if uri.authority().is_some() {
            Self::Authority
        } else if uri.path() == "*" {
            Self::Asterisk
        } else {
            Self::AbsPath
        }
    }
}

impl From<Uri> for RequestUri {
    fn from(uri: Uri) -> Self {
        Self::from(&uri)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_from_uri() {
        assert_eq!(
            RequestUri::from("*".parse::<Uri>().unwrap()),
            RequestUri::Asterisk
        );

        for uri in &[
            "http://www.w3.org/pub/WWW/TheProject.html",
            "http://www.w3.org",
        ] {
            assert_eq!(
                RequestUri::from(uri.parse::<Uri>().unwrap()),
                RequestUri::AbsoluteUri
            );
        }

        for uri in &["/pub/WWW/TheProject.html", "/"] {
            assert_eq!(
                RequestUri::from(uri.parse::<Uri>().unwrap()),
                RequestUri::AbsPath
            );
        }

        assert_eq!(
            RequestUri::from("proxy.com".parse::<Uri>().unwrap()),
            RequestUri::Authority
        );
    }
}
