//! https://datatracker.ietf.org/doc/html/rfc2616#section-5.1.2

use http::Uri;

#[derive(Debug, Copy, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum HttpRequestUri {
    Asterisk,
    AbsoluteUri,
    AbsPath,
    Authority,
}

impl From<&Uri> for HttpRequestUri {
    fn from(uri: &Uri) -> Self {
        if uri.scheme().is_some() {
            Self::AbsoluteUri
        } else if uri.authority().is_some() {
            Self::Authority
        } else if uri.path() == "*" {
            Self::Asterisk
        } else {
            Self::AbsPath
        }
    }
}

impl From<Uri> for HttpRequestUri {
    fn from(uri: Uri) -> Self {
        Self::from(&uri)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_from_uri() {
        assert_eq!(
            HttpRequestUri::from("*".parse::<Uri>().unwrap()),
            HttpRequestUri::Asterisk
        );

        for uri in &[
            "http://www.w3.org/pub/WWW/TheProject.html",
            "http://www.w3.org",
        ] {
            assert_eq!(
                HttpRequestUri::from(uri.parse::<Uri>().unwrap()),
                HttpRequestUri::AbsoluteUri
            );
        }

        for uri in &["/pub/WWW/TheProject.html", "/"] {
            assert_eq!(
                HttpRequestUri::from(uri.parse::<Uri>().unwrap()),
                HttpRequestUri::AbsPath
            );
        }

        assert_eq!(
            HttpRequestUri::from("proxy.com".parse::<Uri>().unwrap()),
            HttpRequestUri::Authority
        );
    }
}
