//! Types to represent the parser errors that can occur.
use std::fmt::Display;

#[derive(PartialEq, Eq, Debug)]
#[non_exhaustive]
pub enum Error {
    AttributesOnEndTag,
    SelfClosingEndTag,
    DuplicateAttribute {
        #[cfg(feature = "spans")]
        #[cfg_attr(docsrs, doc(cfg(feature = "spans")))]
        /// Span of the duplicate attribute name.
        span: std::ops::Range<usize>,
    },
    BadCharacter(char),
    UnexpectedCharacter(char, InternalState),
    UnexpectedEOF(InternalState),
    CharRef(CharRefError),
}

/// Allows Error variants to include the internal tokenizer state without making it public.
#[derive(PartialEq, Eq, Debug)]
pub struct InternalState(pub(crate) crate::tokenizer::states::State);

impl Display for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Error::AttributesOnEndTag => write!(f, "attributes on an end tag"),
            Error::SelfClosingEndTag => write!(f, "self-closing end tag"),
            Error::DuplicateAttribute { .. } => write!(f, "duplicate attribute"),
            Error::BadCharacter(char) => write!(f, "bad character {:?}", char),
            Error::UnexpectedCharacter(char, state) => {
                write!(
                    f,
                    "unexpected character: saw {:?} in state {:?}",
                    char, state.0
                )
            }
            Error::UnexpectedEOF(state) => write!(f, "unexpected EOF in state {:?}", state.0),
            Error::CharRef(error) => error.fmt(f),
        }
    }
}

#[derive(PartialEq, Eq, Debug)]
#[non_exhaustive]
pub enum CharRefError {
    MissingSemicolon,
    NumericCharRefWithoutDigits,
    NumericCharRefInvalid(u32),
    EofInNumericCharRef,
    EofAfterNumberSign,
    EqualsSignAfterCharRefInAttr,
    #[cfg(feature = "named-entities")]
    #[cfg_attr(docsrs, doc(cfg(feature = "named-entities")))]
    InvalidNamedCharRef,
}

impl Display for CharRefError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            CharRefError::NumericCharRefWithoutDigits => {
                write!(f, "numeric character reference without digits")
            }
            CharRefError::MissingSemicolon => {
                write!(f, "semicolon missing after character reference")
            }
            CharRefError::NumericCharRefInvalid(num) => {
                write!(f, "invalid numeric character reference value 0x{:06X}", num)
            }
            CharRefError::EofInNumericCharRef => {
                write!(f, "EOF in numeric character reference")
            }
            CharRefError::EofAfterNumberSign => {
                write!(f, "EOF after '#' in character reference")
            }
            CharRefError::EqualsSignAfterCharRefInAttr => {
                write!(f, "equals sign after character reference in attribute")
            }
            #[cfg(feature = "named-entities")]
            CharRefError::InvalidNamedCharRef => {
                write!(f, "invalid named character reference")
            }
        }
    }
}
