// Copyright 2014-2017 The html5ever Project Developers. See the
// COPYRIGHT file at the top-level directory of this distribution.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

#[cfg(feature = "named-entities")]
extern crate phf_codegen;

#[cfg(feature = "named-entities")]
use {std::collections::HashMap, std::env, std::fs::File, std::io::Write, std::path::Path};

#[cfg(feature = "named-entities")]
mod entities;

fn main() {
    #[cfg(feature = "named-entities")]
    named_entities_to_phf(&Path::new(&env::var("OUT_DIR").unwrap()).join("named_entities.rs"));
}

#[cfg(feature = "named-entities")]
fn named_entities_to_phf(to: &Path) {
    let mut entities: HashMap<&str, (u32, u32)> = entities::NAMED_ENTITIES
        .iter()
        .map(|(name, cp1, cp2)| {
            assert!(name.starts_with('&'));
            (&name[1..], (*cp1, *cp2))
        })
        .collect();

    // Add every missing prefix of those keys, mapping to NULL characters.
    for key in entities.keys().cloned().collect::<Vec<_>>() {
        for n in 1..key.len() {
            entities.entry(&key[..n]).or_insert((0, 0));
        }
    }
    entities.insert("", (0, 0));

    let mut phf_map = phf_codegen::Map::new();
    for (key, value) in entities {
        phf_map.entry(key, &format!("{:?}", value));
    }

    let mut file = File::create(to).unwrap();
    writeln!(
        &mut file,
        r#"
/// A map of entity names to their codepoints. The second codepoint will
/// be 0 if the entity contains a single codepoint. Entities have their preceeding '&' removed.
"#
    )
    .unwrap();
    writeln!(
        &mut file,
        "pub static NAMED_ENTITIES: Map<&'static str, (u32, u32)> = {};",
        phf_map.build(),
    )
    .unwrap();
}
