// SPDX-FileCopyrightText: 2021 Carson Black <uhhadd@gmail.com>
//
// SPDX-License-Identifier: MPL-2.0

package main

import (
	"fmt"

	"google.golang.org/protobuf/compiler/protogen"
)

func GenerateGoClient(gen *protogen.Plugin, file *protogen.File) {
	if len(file.Services) == 0 {
		return
	}

	filename := file.GeneratedFilenamePrefix + "_hrpc_client.pb.go"
	g := gen.NewGeneratedFile(filename, file.GoImportPath)

	g.P("// Code generated by protoc-gen-go-hrpc. DO NOT EDIT.")
	g.P()
	g.P("package ", file.GoPackageName)
	g.P()
	genClientContent(g, file)
}

func genClientContent(g *protogen.GeneratedFile, file *protogen.File) {
	for _, service := range file.Services {
		genClientService(g, service)
	}
}

func genClientService(g *protogen.GeneratedFile, service *protogen.Service) {
	serverType := service.GoName + "Client"
	g.P("type ", serverType, " interface {")
	for _, method := range service.Methods {
		g.P(method.Comments.Leading, serverSignature(g, method))
	}
	g.P("}")
	g.P()
	genClientHTTPImpl(g, service)
	genClientTestHTTPImpl(g, service)
}

func genClientHTTPImpl(g *protogen.GeneratedFile, service *protogen.Service) {
	dummyType := "HTTP" + service.GoName + "Client"
	g.P("type ", dummyType, " struct {")
	g.P("Client ", g.QualifiedGoIdent(httpPackage.Ident("Client")))
	g.P("BaseURL string")
	g.P("}")
	for _, method := range service.Methods {
		g.P("func (", "client *", dummyType, ") ", clientSignature(g, method), " {")
		if method.Desc.IsStreamingClient() && method.Desc.IsStreamingServer() {
			g.P("return nil, ", g.QualifiedGoIdent(errorPackage.Ident("New")), `("unimplemented")`)
		} else if method.Desc.IsStreamingServer() {
			g.P("return nil, ", g.QualifiedGoIdent(errorPackage.Ident("New")), `("unimplemented")`)
		} else if method.Desc.IsStreamingClient() {
			g.P("return nil, ", g.QualifiedGoIdent(errorPackage.Ident("New")), `("unimplemented")`)
		} else {
			genClientUnary(g, service, method)
		}

		g.P("}")
	}
}

func genClientTestHTTPImpl(g *protogen.GeneratedFile, service *protogen.Service) {
	dummyType := "HTTPTest" + service.GoName + "Client"
	g.P("type ", dummyType, " struct {")
	g.P("Client ", "interface { Test(", "*", g.QualifiedGoIdent(httpPackage.Ident("Request")), ", ...int", ")", "(*", httpPackage.Ident("Response"), ", error) }")
	g.P("}")
	for _, method := range service.Methods {
		g.P("func (", "client *", dummyType, ") ", clientSignature(g, method), " {")
		if method.Desc.IsStreamingClient() && method.Desc.IsStreamingServer() {
			g.P("return nil, ", g.QualifiedGoIdent(errorPackage.Ident("New")), `("unimplemented")`)
		} else if method.Desc.IsStreamingServer() {
			g.P("return nil, ", g.QualifiedGoIdent(errorPackage.Ident("New")), `("unimplemented")`)
		} else if method.Desc.IsStreamingClient() {
			g.P("return nil, ", g.QualifiedGoIdent(errorPackage.Ident("New")), `("unimplemented")`)
		} else {
			genClientTestUnary(g, service, method)
		}

		g.P("}")
	}
}

func genClientMarshal(fromVar, toVar, err string, g *protogen.GeneratedFile, service *protogen.Service) {
	marshal := g.QualifiedGoIdent(protoPackage.Ident("Marshal"))

	g.P(toVar, ", ", err, " := ", marshal, "(", fromVar, ")")
	g.P("if ", err, "!=", "nil", "{")
	g.P("return nil, ", err)
	g.P("}")
}

func genClientUnmarshal(fromVar, toVar, err string, kind protogen.GoIdent, g *protogen.GeneratedFile, service *protogen.Service) {
	unmarshal := g.QualifiedGoIdent(protoPackage.Ident("Unmarshal"))

	g.P(toVar, " := ", "&", g.QualifiedGoIdent(kind), "{}")
	g.P(err, " := ", unmarshal, "(", fromVar, ",", toVar, ")")
	g.P("if ", err, "!=", "nil", "{")
	g.P("return nil, ", err)
	g.P("}")
}

func genClientUnary(g *protogen.GeneratedFile, service *protogen.Service, method *protogen.Method) {
	genClientMarshal("req", "data", "marshalErr", g, service)
	g.P("reader := ", g.QualifiedGoIdent(bytesPackage.Ident("NewReader")), "(data)")
	g.P("resp, err := ", "client.Client.Post(client.BaseURL + ", fmt.Sprintf(`"/%s/"`, method.Desc.FullName()), `, "application/hrpc", reader)`)
	g.P("if err != nil {")
	g.P("return nil, err")
	g.P("}")
	g.P("body, err := ", g.QualifiedGoIdent(ioutilPackage.Ident("ReadAll")), "(resp.Body)")
	g.P("if err != nil {")
	g.P("return nil, err")
	g.P("}")
	genClientUnmarshal("body", "ret", "unmarshalErr", method.Output.GoIdent, g, service)
	g.P("return ret, nil")
}

func genClientTestUnary(g *protogen.GeneratedFile, service *protogen.Service, method *protogen.Method) {
	genClientMarshal("req", "data", "marshalErr", g, service)
	g.P("reader := ", g.QualifiedGoIdent(bytesPackage.Ident("NewReader")), "(data)")
	g.P("testreq := ", g.QualifiedGoIdent(httptestPackage.Ident("NewRequest")), `("POST", `, fmt.Sprintf(`"/%s/"`, method.Desc.FullName()), `, reader)`)
	g.P("resp, err := client.Client.Test(testreq)")
	g.P("if err != nil {")
	g.P("return nil, err")
	g.P("}")
	g.P("body, err := ", g.QualifiedGoIdent(ioutilPackage.Ident("ReadAll")), "(resp.Body)")
	g.P("if err != nil {")
	g.P("return nil, err")
	g.P("}")
	genClientUnmarshal("body", "ret", "unmarshalErr", method.Output.GoIdent, g, service)
	g.P("return ret, nil")
}

func clientSignature(g *protogen.GeneratedFile, m *protogen.Method) string {
	var inputArg string
	var ret string
	inputType := g.QualifiedGoIdent(m.Input.GoIdent)
	outputType := g.QualifiedGoIdent(m.Output.GoIdent)
	if m.Desc.IsStreamingClient() {
		inputArg = "chan *" + inputType
	} else {
		inputArg = "*" + inputType
	}
	if m.Desc.IsStreamingServer() {
		ret = "chan *" + outputType
	} else {
		ret = "*" + outputType
	}
	return fmt.Sprintf("%s(req %s) (%s, error)", m.GoName, inputArg, ret)
}
