//! This file contains lib to call hnsw from julia (or any language providing a C api)
//! The AnnT trait is implemented with macros for u32, u16, u8, f32, f64 and i32.  
//! The macro declare_myapi_type!  produces struct HnswApif32 and so on.
//! 

use std::ptr;
use std::fs::OpenOptions;
use std::io::{BufReader};
use std::path::{PathBuf};

use log;

use crate::api::*;
use crate::dist::*;
use crate::hnsw::*;
use crate::hnswio::*;



// the export macro makes the macro global in crate and accecssible via crate::declare_myapi_type!
#[macro_export]
macro_rules! declare_myapi_type(
    ($name:ident, $ty:ty) => (
        pub struct $name {
#[allow(dead_code)]
        pub(crate) opaque: Box<dyn AnnT<Val=$ty>>,
        } // end struct
        impl $name {
            pub fn new(arg: Box<dyn AnnT<Val=$ty>>) -> Self {
                $name{ opaque:arg}
            } // end new
        } // end impl
    )
);


declare_myapi_type!(HnswApiNodata, NoData);

// declare_myapi_type!(HnswApif64, f64);
// declare_myapi_type!(HnswApif32, f32);

/// to be able to return a vector from rust in a julia struct before converting to a julia Vector
#[repr(C)]
pub struct Vec_api<T> {
    len : i64,
    ptr : *const T,
} // end struct



#[repr(C)]
/// The basic Neighbour info returned by api
pub struct Neighbour_api {
    /// id of neighbour
    pub id  : usize,
    /// distance of data sent in request to this neighbour
    pub d   : f32,
}


impl From<&Neighbour> for Neighbour_api {
    fn from(neighbour : &Neighbour) -> Self {
        Neighbour_api{ id : neighbour.d_id, d: neighbour.distance}
    }
}


#[repr(C)]
/// The response to a neighbour search requests
pub struct Neighbourhood_api {
    pub nbgh : i64,
    pub neighbours : *const Neighbour_api,
}



#[repr(C)]
pub struct Neighbour_api_parsearch_answer {
    /// The number of answers (o request), i.e size of both vectors nbgh and neighbours
    pub nb_answer : usize,
    /// for each request, we get a Neighbourhood_api
    pub neighbourhoods : *const Neighbourhood_api,
}

//===================================== f32  type =====================================


// macros have been exported to the root of the crate so we do not refer to them via api::
super::declare_myapi_type!(HnswApif32, f32);
super::declare_myapi_type!(HnswApif64, f64);



//===================================================================================================
//  These are the macros to generate trait implementation for useful numeric types
#[allow(unused_macros)]

macro_rules! generate_insert(
($function_name:ident, $api_name:ty, $type_val:ty) => (
        #[no_mangle]
        pub extern "C" fn $function_name(hnsw_api : *mut $api_name, len:usize, data : *const $type_val, id : usize) {
        log::trace!("entering insert, type {:?} vec len is {:?}, id : {:?} ", stringify!($type_val), len, id);
        //  construct vector: Rust clones and take ownership.
        let data_v : Vec<$type_val>;
        unsafe {
            let slice = std::slice::from_raw_parts(data, len);
            data_v = Vec::from(slice);
            log::trace!("calling insert data");
            (*hnsw_api).opaque.insert_data(&data_v, id);
        }
        log::trace!("exiting insert for type {:?}", stringify!($type_val));
        }  // end of insert
    )
);



macro_rules! generate_parallel_insert(
($function_name:ident, $api_name:ty, $type_val:ty) => (
        #[no_mangle]
        pub extern "C" fn $function_name(hnsw_api : *mut $api_name, nb_vec: usize, vec_len : usize, 
                        datas : *mut *const $type_val, ids : *const usize) {
            //
            log::trace!("entering parallel_insert type {:?}  , vec len is {:?}, nb_vec : {:?}", stringify!($type_val), vec_len, nb_vec);
            let data_ids : Vec<usize>;
            let data_ptrs : Vec<*const $type_val>;
            unsafe {
                let slice = std::slice::from_raw_parts(ids, nb_vec);
                data_ids = Vec::from(slice);
            }
            // log::debug!("got ids");
            unsafe {
                let slice = std::slice::from_raw_parts(datas, nb_vec);
                data_ptrs = Vec::from(slice);
            }
            // log::debug!("got data ptrs");
            let mut data_v = Vec::<Vec<$type_val>>::with_capacity(nb_vec);
            for i in 0..nb_vec {
                unsafe {
                    let slice = std::slice::from_raw_parts(data_ptrs[i], vec_len);
                    let v = Vec::from(slice);
                    data_v.push(v);
                }
            }
            // log::debug!("sending request");
            let mut request : Vec<(&Vec<$type_val>, usize)> = Vec::with_capacity(nb_vec);
            for i in 0..nb_vec {
                request.push((&data_v[i], data_ids[i]));
            }
            //
            unsafe { (*hnsw_api).opaque.parallel_insert_data(&request); };
            log::trace!("exiting parallel_insert");
        } // end of parallel_insert
    )   
);


macro_rules! generate_search_neighbours(
($function_name:ident, $api_name:ty, $type_val:ty) => (
        #[no_mangle]
        pub extern "C" fn $function_name(hnsw_api : *const $api_name, len:usize, data : *const $type_val, 
                                knbn : usize, ef_search : usize) ->  *const Neighbourhood_api {
            //                       
            log::trace!("entering search_neighbours type {:?}, vec len is {:?}, id : {:?} ef_search {:?}", stringify!($type_val), len, knbn, ef_search);
            let data_v : Vec<$type_val>;
            let neighbours : Vec<Neighbour>; 
            unsafe {
                let slice = std::slice::from_raw_parts(data, len);
                data_v = Vec::from(slice);
                log::trace!("calling search neighbours {:?}", data_v);
                neighbours =  (*hnsw_api).opaque.search_neighbours(&data_v, knbn, ef_search);
            }
            let neighbours_api : Vec<Neighbour_api> = neighbours.iter().map(|n| Neighbour_api::from(n)).collect();
            log::trace!(" got nb neighbours {:?}", neighbours_api.len());
            // for i in 0..neighbours_api.len() {
            //    println!(" id {:?}  dist : {:?} ", neighbours_api[i].id, neighbours_api[i].d);
            // }    
            let nbgh_i = neighbours.len() as i64;
            let neighbours_ptr = neighbours_api.as_ptr();
            std::mem::forget(neighbours_api);
            let answer = Neighbourhood_api {
                    nbgh : nbgh_i,
                    neighbours : neighbours_ptr,
            };
            log::trace!("search_neighbours returning nb neighbours {:?} id ptr {:?} ", nbgh_i, neighbours_ptr);
            Box::into_raw(Box::new(answer))
        }
    )
);




macro_rules! generate_parallel_search_neighbours(
($function_name:ident, $api_name:ty, $type_val:ty) => (
        #[no_mangle]
        /// search nb_vec of size vec_len. The the searches will be done in // as far as possible.
        pub extern "C" fn $function_name(hnsw_api : *const $api_name, nb_vec : usize, vec_len :i64, 
                            data : *mut *const $type_val, knbn : usize, ef_search : usize) ->  *const Vec_api<Neighbourhood_api> {
            //
            // must build a Vec<Vec<f32> to build request
            log::trace!("recieving // search request for type: {:?} with {:?} vectors", stringify!($type_val), nb_vec);
            let neighbours : Vec<Vec<Neighbour> >; 
            let mut data_v = Vec::<Vec<$type_val>>::with_capacity(nb_vec);
            unsafe {
                let slice = std::slice::from_raw_parts(data, nb_vec);
                let ptr_list : Vec<*const $type_val> = Vec::from(slice);
                for i in 0..nb_vec {
                    let slice_i = std::slice::from_raw_parts(ptr_list[i], vec_len as usize);
                    let v = Vec::from(slice_i);
                    data_v.push(v);
                }
            // log::debug!(" reconstructed input vectors");
            neighbours =  (*hnsw_api).opaque.parallel_search_neighbours(&data_v, knbn, ef_search);
            }
            // construct a vector of Neighbourhood_api
            // reverse work, construct 2 arrays, one vector of Neighbours, and one vectors of number of returned neigbours by input a vector.
            let mut neighbour_lists = Vec::<Neighbourhood_api>::with_capacity(nb_vec);
            for v in neighbours {
                let neighbours_api : Vec<Neighbour_api> = v.iter().map(|n| Neighbour_api::from(n)).collect();
                let nbgh = neighbours_api.len();
                let neighbours_api_ptr = neighbours_api.as_ptr();
                std::mem::forget(neighbours_api);
                let v_answer = Neighbourhood_api {
                        nbgh : nbgh as i64,
                        neighbours: neighbours_api_ptr,
                    };
                neighbour_lists.push(v_answer);
            }
            log::trace!(" reconstructed output  pointers to vectors");
            let neighbour_lists_ptr = neighbour_lists.as_ptr();
            std::mem::forget(neighbour_lists);
            let answer = Vec_api::<Neighbourhood_api> {
                len : nb_vec as i64,
                ptr : neighbour_lists_ptr,
            };
            Box::into_raw(Box::new(answer))
        }  // end of parallel_search_neighbours_f32 for HnswApif32
    )
);

#[allow(unused_macros)]
macro_rules! generate_file_dump(
    ($function_name:ident, $api_name:ty, $type_val:ty) => (
    #[no_mangle]
        pub extern "C" fn $function_name(hnsw_api : *const $api_name, namelen : usize, filename :*const u8) -> i64 {
            log::info!("receiving request for file dump");
            let slice = unsafe { std::slice::from_raw_parts(filename, namelen) } ;
            let fstring  = String::from_utf8_lossy(slice).into_owned();
            let res =  unsafe { (*hnsw_api).opaque.file_dump(&fstring) } ;
            if res.is_ok() {
                return 1;
            }
            else { return -1; }
        } // end of function_name
    )
);


#[allow(unused_macros)]
macro_rules! generate_loadhnsw(
    ($function_name:ident, $api_name:ty, $type_val:ty, $type_dist : ty) => (
        #[no_mangle]
        pub extern "C" fn $function_name(flen : usize, name : *const u8)  -> *const $api_name {
            let  slice = unsafe { std::slice::from_raw_parts(name, flen)} ;
            let filename = String::from_utf8_lossy(slice).into_owned(); 
            //
            let buffers = make_readers(&filename);
            let mut graph_in = buffers.0;
            let mut data_in = buffers.1;
            // we need to call load_description first to get distance name
            let hnsw_description = load_description(&mut graph_in).unwrap();
                // here we need a macro to dispatch 
            let hnsw_loaded_res = load_hnsw::<$type_val, $type_dist>(&mut graph_in, &hnsw_description, &mut data_in);
            if let Ok(hnsw_loaded) = hnsw_loaded_res {
                let api = <$api_name>::new(Box::new(hnsw_loaded));
                return Box::into_raw(Box::new(api));  
            }
            else {
                log::warn!("an error occured, could not reload data from {:?}", filename);
            }
            return ptr::null();
        }  // end of load_hnswdump_   
     )
);

// here we must generate as many function as there are couples (type, distance) to be accessed from our needs in Julia

// f32
generate_loadhnsw!(load_hnswdump_f32_DistL1, HnswApif32, f32, crate::dist::DistL1);
generate_loadhnsw!(load_hnswdump_f32_DistL2, HnswApif32, f32, crate::dist::DistL2);
generate_loadhnsw!(load_hnswdump_f32_DistCosine, HnswApif32, f32, crate::dist::DistCosine);
generate_loadhnsw!(load_hnswdump_f32_DistDot, HnswApif32, f32, crate::dist::DistDot);
generate_loadhnsw!(load_hnswdump_f32_DistJensenShannon, HnswApif32, f32, crate::dist::DistJensenShannon);
generate_loadhnsw!(load_hnswdump_f32_DistJeffreys, HnswApif32, f32, crate::dist::DistJeffreys);

// i32
generate_loadhnsw!(load_hnswdump_i32_DistL1, HnswApii32, i32, crate::dist::DistL1);
generate_loadhnsw!(load_hnswdump_i32_DistL2, HnswApii32, i32, crate::dist::DistL2);
generate_loadhnsw!(load_hnswdump_i32_DistHamming, HnswApii32, i32, crate::dist::DistHamming);

// u32
generate_loadhnsw!(load_hnswdump_u32_DistL1, HnswApiu32, u32, crate::dist::DistL1);
generate_loadhnsw!(load_hnswdump_u32_DistL2, HnswApiu32, u32, crate::dist::DistL2);
generate_loadhnsw!(load_hnswdump_u32_DistHamming, HnswApiu32, u32, crate::dist::DistHamming);
generate_loadhnsw!(load_hnswdump_u32_DistJaccard, HnswApiu32, u32, crate::dist::DistJaccard);

// u16
generate_loadhnsw!(load_hnswdump_u16_DistL1, HnswApiu16, u16, crate::dist::DistL1);
generate_loadhnsw!(load_hnswdump_u16_DistL2, HnswApiu16, u16, crate::dist::DistL2);
generate_loadhnsw!(load_hnswdump_u16_DistHamming, HnswApiu16, u16, crate::dist::DistHamming);
generate_loadhnsw!(load_hnswdump_u16_DistLevenshtein, HnswApiu16, u16, crate::dist::DistLevenshtein);

// u8
generate_loadhnsw!(load_hnswdump_u8_DistL1, HnswApiu8, u8, crate::dist::DistL1);
generate_loadhnsw!(load_hnswdump_u8_DistL2, HnswApiu8, u8, crate::dist::DistL2);
generate_loadhnsw!(load_hnswdump_u8_DistHamming, HnswApiu8, u8, crate::dist::DistHamming);
generate_loadhnsw!(load_hnswdump_u8_DistJaccard, HnswApiu8, u8, crate::dist::DistJaccard);


// Reload only graph
generate_loadhnsw!(load_hnswdump_NoData_DistNoDist, HnswApiNodata, NoData, crate::dist::NoDist);

//=============== implementation for i32



#[no_mangle]
pub extern "C" fn init_hnsw_f32(max_nb_conn : usize, ef_const:usize, namelen: usize,  cdistname : *const u8) -> *const HnswApif32 {
    log::info!("entering init_hnsw_f32");
    let  slice = unsafe { std::slice::from_raw_parts(cdistname, namelen)} ;
    let dname = String::from_utf8_lossy(slice).into_owned();
    // map distname to sthg. This whole block will go to a macro
    match dname.as_str() {
    "DistL1" => { 
        log::info!(" received DistL1");
        let h = Hnsw::<f32, DistL1 >::new(max_nb_conn, 10000, 16, ef_const, DistL1{});
        let api = HnswApif32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));
    }
    "DistL2" => {
        let h = Hnsw::<f32, DistL2 >::new(max_nb_conn, 10000, 16, ef_const, DistL2{});
        let api = HnswApif32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));        
    }
    "DistDot" =>  {
        let h = Hnsw::<f32, DistDot >::new(max_nb_conn, 10000, 16, ef_const, DistDot{});
        let api = HnswApif32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));        
    }
    "DistHellinger" =>  {
        let h = Hnsw::<f32, DistHellinger >::new(max_nb_conn, 10000, 16, ef_const, DistHellinger{});
        let api = HnswApif32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));        
    }
    "DistJeffreys" =>  {
        let h = Hnsw::<f32, DistJeffreys>::new(max_nb_conn, 10000, 16, ef_const, DistJeffreys{});
        let api = HnswApif32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));        
    }
    "DistJensenShannon" =>  {
        let h = Hnsw::<f32, DistJensenShannon>::new(max_nb_conn, 10000, 16, ef_const, DistJensenShannon{});
        let api = HnswApif32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));        
    }
    _   => {
        log::warn!("init_hnsw_f32 received unknow distance {:?} ", dname);
        let p = ptr::null::< HnswApif32 >();
        return p;
    }
    } // znd match
} // end of init_hnsw_f32



#[no_mangle]
pub extern "C" fn init_hnsw_ptrdist_f32(max_nb_conn : usize, ef_const:usize, c_func : extern "C" fn(*const f32, *const f32, u64) -> f32 ) -> *const HnswApif32 {
    log::info!("init_ hnsw_ptrdist: ptr  {:?}", c_func);
    let c_dist = DistCFFI::<f32>::new(c_func);
    let h = Hnsw::<f32, DistCFFI<f32> >::new(max_nb_conn, 10000, 16, ef_const, c_dist);
    let api = HnswApif32{opaque: Box::new(h)};
    return Box::into_raw(Box::new(api));
}

#[no_mangle]
pub extern "C" fn insert_f32(hnsw_api : *mut HnswApif32, len:usize, data : *const f32, id : usize) {
    log::trace!("entering insert_f32, vec len is {:?}, id : {:?} ", len, id);
    //  construct vector: Rust clones and take ownership.
    let data_v : Vec<f32>;
    unsafe {
        let slice = std::slice::from_raw_parts(data, len);
        data_v = Vec::from(slice);
    //    log::debug!("calling insert data");
        (*hnsw_api).opaque.insert_data(&data_v, id);
    }
    // log::trace!("exiting insert_f32");
}  // end of insert_f32


#[no_mangle]
pub extern "C" fn parallel_insert_f32(hnsw_api : *mut HnswApif32, nb_vec: usize, vec_len : usize, 
                datas : *mut *const f32, ids : *const usize) {
    //
    // log::debug!("entering parallel_insert_f32 , vec len is {:?}, nb_vec : {:?}", vec_len, nb_vec);
    let data_ids : Vec<usize>;
    let data_ptrs : Vec<*const f32>;
    unsafe {
        let slice = std::slice::from_raw_parts(ids, nb_vec);
        data_ids = Vec::from(slice);
    }
    unsafe {
        let slice = std::slice::from_raw_parts(datas, nb_vec);
        data_ptrs = Vec::from(slice);
    }
    // log::debug!("got data ptrs");
    let mut data_v = Vec::<Vec<f32>>::with_capacity(nb_vec);
    for i in 0..nb_vec {
        unsafe {
            let slice = std::slice::from_raw_parts(data_ptrs[i], vec_len);
            let v = Vec::from(slice);
            data_v.push(v);
        }
    }
    // log::debug!("sending request");
    let mut request : Vec<(&Vec<f32>, usize)> = Vec::with_capacity(nb_vec);
    for i in 0..nb_vec {
        request.push((&data_v[i], data_ids[i]));
    }
    //
    unsafe { (*hnsw_api).opaque.parallel_insert_data(&request); };
    log::trace!("exiting parallel_insert");
} // end of parallel_insert_f32




#[no_mangle]
pub extern "C" fn search_neighbours_f32(hnsw_api : *const HnswApif32, len:usize, data : *const f32, 
                        knbn : usize, ef_search : usize) ->  *const Neighbourhood_api {
    //                       
    log::trace!("entering search_neighbours , vec len is {:?}, id : {:?} ef_search {:?}", len, knbn, ef_search);
    let data_v : Vec<f32>;
    let neighbours : Vec<Neighbour>; 
    unsafe {
        let slice = std::slice::from_raw_parts(data, len);
        data_v = Vec::from(slice);
        log::trace!("calling search neighbours {:?}", data_v);
        neighbours =  (*hnsw_api).opaque.search_neighbours(&data_v, knbn, ef_search);
    }
    let neighbours_api : Vec<Neighbour_api> = neighbours.iter().map(|n| Neighbour_api::from(n)).collect();
    log::trace!(" got nb neighbours {:?}", neighbours_api.len());
    // for i in 0..neighbours_api.len() {
    //    println!(" id {:?}  dist : {:?} ", neighbours_api[i].id, neighbours_api[i].d);
    // }    
    let nbgh_i = neighbours.len() as i64;
    let neighbours_ptr = neighbours_api.as_ptr();
    std::mem::forget(neighbours_api);
    let answer = Neighbourhood_api {
            nbgh : nbgh_i,
            neighbours : neighbours_ptr,
    };
    log::trace!("search_neighbours returning nb neighbours {:?} id ptr {:?} ", nbgh_i, neighbours_ptr);
    Box::into_raw(Box::new(answer))
}
  // end of search_neighbours for HnswApif32




generate_parallel_search_neighbours!(parallel_search_neighbours_f32, HnswApif32, f32);
generate_file_dump!(file_dump_f32, HnswApif32, f32);



//=============== implementation for i32

#[no_mangle]
pub extern "C" fn init_hnsw_i32(max_nb_conn : usize, ef_const:usize, namelen: usize,  cdistname : *const u8) -> *const HnswApii32 {
    log::info!("entering init_hnsw_i32");
    let  slice = unsafe { std::slice::from_raw_parts(cdistname, namelen)} ;
    let dname = String::from_utf8_lossy(slice);
    // map distname to sthing. This whole block will go to a macro
    if dname == "DistL1" { 
        log::info!(" received DistL1");
        let h = Hnsw::<i32, DistL1 >::new(max_nb_conn, 10000, 16, ef_const, DistL1{});
        let api = HnswApii32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));
    }
    else if dname == "DistL2" {
        let h = Hnsw::<i32, DistL2 >::new(max_nb_conn, 10000, 16, ef_const, DistL2{});
        let api = HnswApii32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));        
    }
    else if dname == "DistHamming" {
        let h = Hnsw::<i32, DistHamming>::new(max_nb_conn, 10000, 16, ef_const, DistHamming{});
        let api = HnswApii32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));         
    } 
    let p = ptr::null::< HnswApii32 >();
    p
} // end of init_hnsw_i32




#[no_mangle]
pub extern "C" fn init_hnsw_ptrdist_i32(max_nb_conn : usize, ef_const:usize, 
                c_func : extern "C" fn(*const i32, *const i32, u64) -> f32 ) -> *const HnswApii32 {
    log::debug!("init_ hnsw_ptrdist: ptr  {:?}", c_func);
    let c_dist = DistCFFI::<i32>::new(c_func);
    let h = Hnsw::<i32, DistCFFI<i32> >::new(max_nb_conn, 10000, 16, ef_const, c_dist);
    let api = HnswApii32{opaque: Box::new(h)};
    return Box::into_raw(Box::new(api));
}


//==generation of function for i32

super::declare_myapi_type!(HnswApii32, i32);

generate_insert!(insert_i32, HnswApii32, i32);
generate_parallel_insert!(parallel_insert_i32, HnswApii32, i32);
generate_search_neighbours!(search_neighbours_i32, HnswApii32, i32);
generate_parallel_search_neighbours!(parallel_search_neighbours_i32, HnswApii32, i32);
generate_file_dump!(file_dump_i32, HnswApii32, i32);

//========== generation for u32


#[no_mangle]
pub extern "C" fn init_hnsw_u32(max_nb_conn : usize, ef_const:usize, namelen: usize,  cdistname : *const u8) -> *const HnswApiu32 {
    log::debug!("entering init_hnsw_u32");
    let  slice = unsafe { std::slice::from_raw_parts(cdistname, namelen)} ;
    let dname = String::from_utf8_lossy(slice);
    // map distname to sthg. This whole block will go to a macro
    if dname == "DistL1" { 
        log::debug!(" received DistL1");
        let h = Hnsw::<u32, DistL1 >::new(max_nb_conn, 10000, 16, ef_const, DistL1{});
        let api = HnswApiu32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));
    }
    else if dname == "DistL2" {
        let h = Hnsw::<u32, DistL2 >::new(max_nb_conn, 10000, 16, ef_const, DistL2{});
        let api = HnswApiu32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api)); 
    }       
    else if dname == "DistJaccard" {
        let h = Hnsw::<u32, DistJaccard >::new(max_nb_conn, 10000, 16, ef_const, DistJaccard{});
        let api = HnswApiu32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));         
    }
    else if dname == "DistHamming" {
        let h = Hnsw::<u32, DistHamming>::new(max_nb_conn, 10000, 16, ef_const, DistHamming{});
        let api = HnswApiu32{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));         
    } 
    //
    let p = ptr::null::< HnswApiu32 >();
    p
} // end of init_hnsw_i32



#[no_mangle]
pub extern "C" fn init_hnsw_ptrdist_u32(max_nb_conn : usize, ef_const:usize, 
                c_func : extern "C" fn(*const u32, *const u32, u64) -> f32 ) -> *const HnswApiu32 {
    log::info!("init_ hnsw_ptrdist: ptr  {:?}", c_func);
    let c_dist = DistCFFI::<u32>::new(c_func);
    let h = Hnsw::<u32, DistCFFI<u32> >::new(max_nb_conn, 10000, 16, ef_const, c_dist);
    let api = HnswApiu32{opaque: Box::new(h)};
    return Box::into_raw(Box::new(api));
}


super::declare_myapi_type!(HnswApiu32, u32);

generate_insert!(insert_u32, HnswApiu32, u32);
generate_parallel_insert!(parallel_insert_u32, HnswApiu32, u32);
generate_search_neighbours!(search_neighbours_u32, HnswApiu32, u32);
generate_parallel_search_neighbours!(parallel_search_neighbours_u32, HnswApiu32, u32);
generate_file_dump!(file_dump_u32, HnswApiu32, u32);



//============== generation of function for u16 =====================

super::declare_myapi_type!(HnswApiu16, u16);


#[no_mangle]
pub extern "C" fn init_hnsw_u16(max_nb_conn : usize, ef_const:usize, namelen: usize,  cdistname : *const u8) -> *const HnswApiu16 {
    log::info!("entering init_hnsw_u16");
    let  slice = unsafe { std::slice::from_raw_parts(cdistname, namelen)} ;
    let dname = String::from_utf8_lossy(slice);
    // map distname to sthg. This whole block will go to a macro
    if dname == "DistL1" { 
        log::info!(" received DistL1");
        let h = Hnsw::<u16, DistL1 >::new(max_nb_conn, 10000, 16, ef_const, DistL1{});
        let api = HnswApiu16{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));
    }
    else if dname == "DistL2" {
        let h = Hnsw::<u16, DistL2 >::new(max_nb_conn, 10000, 16, ef_const, DistL2{});
        let api = HnswApiu16{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));        
    }
    else if dname == "DistHamming" {
        let h = Hnsw::<u16, DistHamming >::new(max_nb_conn, 10000, 16, ef_const, DistHamming{});
        let api = HnswApiu16{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));         
    }
    else if dname == "DistJaccard" {
        let h = Hnsw::<u16, DistJaccard >::new(max_nb_conn, 10000, 16, ef_const, DistJaccard{});
        let api = HnswApiu16{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));         
    }
    let p = ptr::null::< HnswApiu16 >();
    p
} // end of init_hnsw_u16



#[no_mangle]
pub extern "C" fn init_hnsw_ptrdist_u16(max_nb_conn : usize, ef_const:usize, 
                c_func : extern "C" fn(*const u16, *const u16, u64) -> f32 ) -> *const HnswApiu16 {
    log::info!("init_ hnsw_ptrdist: ptr  {:?}", c_func);
    let c_dist = DistCFFI::<u16>::new(c_func);
    let h = Hnsw::<u16, DistCFFI<u16> >::new(max_nb_conn, 10000, 16, ef_const, c_dist);
    let api = HnswApiu16{opaque: Box::new(h)};
    return Box::into_raw(Box::new(api));
}


generate_insert!(insert_u16, HnswApiu16, u16);
generate_parallel_insert!(parallel_insert_u16, HnswApiu16, u16);
generate_search_neighbours!(search_neighbours_u16, HnswApiu16, u16);
generate_parallel_search_neighbours!(parallel_search_neighbours_u16, HnswApiu16, u16);
generate_file_dump!(file_dump_u16, HnswApiu16, u16);



//============== generation of function for u8 =====================

super::declare_myapi_type!(HnswApiu8, u8);


#[no_mangle]
pub extern "C" fn init_hnsw_u8(max_nb_conn : usize, ef_const:usize, namelen: usize,  cdistname : *const u8) -> *const HnswApiu8 {
    log::debug!("entering init_hnsw_u8");
    let  slice = unsafe { std::slice::from_raw_parts(cdistname, namelen)} ;
    let dname = String::from_utf8_lossy(slice);
    // map distname to sthg. This whole block will go to a macro
    if dname == "DistL1" { 
        log::info!(" received DistL1");
        let h = Hnsw::<u8, DistL1 >::new(max_nb_conn, 10000, 16, ef_const, DistL1{});
        let api = HnswApiu8{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));
    }
    else if dname == "DistL2" {
        let h = Hnsw::<u8, DistL2 >::new(max_nb_conn, 10000, 16, ef_const, DistL2{});
        let api = HnswApiu8{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));        
    }
    else if dname == "DistHamming" {
        let h = Hnsw::<u8, DistHamming >::new(max_nb_conn, 10000, 16, ef_const, DistHamming{});
        let api = HnswApiu8{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));         
    }
    else if dname == "DistJaccard" {
        let h = Hnsw::<u8, DistJaccard >::new(max_nb_conn, 10000, 16, ef_const, DistJaccard{});
        let api = HnswApiu8{opaque: Box::new(h)};
        return Box::into_raw(Box::new(api));         
    }
    let p = ptr::null::< HnswApiu8 >();
    p
} // end of init_hnsw_u16



#[no_mangle]
pub extern "C" fn init_hnsw_ptrdist_u8(max_nb_conn : usize, ef_const:usize, 
                c_func : extern "C" fn(*const u8, *const u8, u64) -> f32 ) -> *const HnswApiu8 {
    log::info!("init_ hnsw_ptrdist: ptr  {:?}", c_func);
    let c_dist = DistCFFI::<u8>::new(c_func);
    let h = Hnsw::<u8, DistCFFI<u8> >::new(max_nb_conn, 10000, 16, ef_const, c_dist);
    let api = HnswApiu8{opaque: Box::new(h)};
    return Box::into_raw(Box::new(api));
}

generate_insert!(insert_u8, HnswApiu8, u8);
generate_parallel_insert!(parallel_insert_u8, HnswApiu8, u8);
generate_search_neighbours!(search_neighbours_u8, HnswApiu8, u8);
generate_parallel_search_neighbours!(parallel_search_neighbours_u8, HnswApiu8, u8);
generate_file_dump!(file_dump_u8, HnswApiu8, u8);



//=========================== dump restore functions

/// This structure provides a light description of the graph to be passed to C compatible languages.
#[repr(C)]
pub struct DescriptionFFI {
    ///  value is 1 for Full 0 for Light
    pub dumpmode : u8,
    /// max number of connections in layers != 0
    pub max_nb_connection : u8,
    /// number of observed layers
    pub nb_layer : u8,
    /// search parameter
    pub ef: usize,
    /// total number of points
    pub nb_point: usize,
    /// dimension of data vector
    pub data_dimension : usize,
    /// length and pointer on dist name
    pub distname_len: usize,
    pub distname : *const u8,
    /// T typename
    pub t_name_len : usize,
    pub t_name : *const u8,
}

impl DescriptionFFI {
    pub fn new() -> Self {
        DescriptionFFI {
            dumpmode : 0,
            max_nb_connection : 0,
            nb_layer : 0,
            ef : 0,
            nb_point : 0,
            data_dimension : 0,
            distname_len : 0,
            distname : ptr::null(),
            t_name_len : 0,
            t_name : ptr::null()
        }
    } // end of new
}


/// returns a const pointer to a DescriptionFFI from a dump file, given filename length and pointer (*const u8)
#[no_mangle]
pub extern "C" fn load_hnsw_description(flen : usize, name : *const u8) -> *const DescriptionFFI {
    // opens file
    let  slice = unsafe { std::slice::from_raw_parts(name, flen)} ;
    let filename = String::from_utf8_lossy(slice).into_owned();   
    let fpath = PathBuf::from(filename);
    let fileres = OpenOptions::new().read(true).open(&fpath);
    //
    let mut ffi_description = DescriptionFFI::new();
    match fileres {
        Ok(file) => {
            //
            let mut bufr = BufReader::with_capacity(10000000 , file);
            let res = load_description(&mut bufr);
            if let Ok(description) = res {

                let distname = String::clone(&description.distname);
                let distname_ptr = distname.as_ptr();
                let distname_len = distname.len();
                std::mem::forget(distname);

                let t_name = String::clone(&description.t_name);
                let t_name_ptr = t_name.as_ptr();
                let t_name_len = t_name.len();
                std::mem::forget(t_name);

                ffi_description.dumpmode = 1;        // CAVEAT
                ffi_description.max_nb_connection = description.max_nb_connection;
                ffi_description.nb_layer = description.nb_layer;
                ffi_description.ef = description.ef;
                ffi_description.data_dimension = description.dimension;
                ffi_description.distname_len = distname_len;
                ffi_description.distname = distname_ptr;
                ffi_description.t_name_len = t_name_len;
                ffi_description.t_name = t_name_ptr;
                return Box::into_raw(Box::new(ffi_description));
            }
            else {
                log::error!("could not get descrption of hnsw from file {:?}", fpath.as_os_str());
                println!("could not get descrption of hnsw from file {:?} ", fpath.as_os_str());
                return ptr::null();
                }
            },
        Err(_e) => {
            log::error!("no such file, load_hnsw_description: could not open file {:?}", fpath.as_os_str());
            println!("no such file, load_hnsw_description: could not open file {:?}", fpath.as_os_str());
            return ptr::null();
        }
    }
}  // end of load_hnsw_description


// This function provides the 2 buffers needed to read the dump of a graph
// in file basename generated by 
fn make_readers(basename: &String) -> (BufReader<std::fs::File>, BufReader<std::fs::File>) {
    let mut graphfname = String::clone(basename);
    graphfname.push_str(".hnsw.graph");
    let graphpath = PathBuf::from(graphfname);
    let graphfileres = OpenOptions::new().read(true).open(&graphpath);
    if graphfileres.is_err() {
        println!("make_readers : could not open file {:?}", graphpath.as_os_str());
        std::panic::panic_any("make_readers : could not open file".to_string());            
    }
    let graphfile = graphfileres.unwrap();
    //  
    let mut datafname = String::clone(basename);
    datafname.push_str(".hnsw.data");
    let datapath = PathBuf::from(datafname);
    let datafileres = OpenOptions::new().read(true).open(&datapath);
    if datafileres.is_err() {
        println!("make_readers: could not open file {:?}", datapath.as_os_str());
        std::panic::panic_any("make_readers: could not open file".to_string());            
    }
    let datafile = datafileres.unwrap();
    //
    let graph_in = BufReader::new(graphfile);
    let data_in = BufReader::new(datafile);
    (graph_in, data_in)
}


//============ log initialization ============//

/// to initialize rust logging from Julia
#[no_mangle]
pub extern "C" fn init_rust_log() {
    let _res = env_logger::Builder::from_default_env().try_init();
}