use crate::utils::*;
use crate::consts::*;
use lazy_static::lazy_static;

lazy_static! {
    static ref LATEX_SYMBOLS: Vec<Vec<u16>> = {
        LATEX_SYMBOLS_RAW.clone().into_iter().map(into_v16).collect()
    };

    static ref LATEX_1ARG_FUNCS: Vec<Vec<u16>> = {
        LATEX_1ARG_FUNCS_RAW.clone().into_iter().map(into_v16).collect()
    };

    static ref LATEX_2ARG_FUNCS: Vec<Vec<u16>> = {
        LATEX_2ARG_FUNCS_RAW.clone().into_iter().map(into_v16).collect()
    };
}


pub fn render_math_formula(content: &Vec<u16>) -> Vec<u16> {

    let mut result = Vec::with_capacity(content.len());
    let mut tmp = vec![];
    let mut index = 0;
    let mut content = content.clone();
    content.push(U16_SPACE);  // so that it handles the last word

    while index < content.len() {

        if is_alphabet(content[index]) {
            tmp.push(content[index]);
        }

        else {

            if tmp.len() > 0 {

                if LATEX_SYMBOLS.contains(&tmp) {

                    if tmp == into_v16("leftcurlybrace") {
                        result.push(into_v16("\\{ "));
                    }

                    else if tmp == into_v16("leftcurlybrace") {
                        result.push(into_v16("\\} "));}

                    else {
                        result.push(
                            vec![
                                vec![U16_BACKSLASH],
                                tmp,
                                vec![U16_SPACE]
                            ].concat()
                        );
                    }
                }

                else if LATEX_1ARG_FUNCS.contains(&tmp) {
                    let (arguments, indexes) = get_arguments(&content, index, vec![], vec![]);

                    if arguments.len() > 0 {

                        let func_name = if tmp == into_v16("sup") {
                            vec![U16_CARET]
                        }

                        else if tmp == into_v16("sub") {
                            vec![U16_UNDERBAR]
                        }

                        else {
                            vec![
                                vec![U16_BACKSLASH],
                                tmp
                            ].concat()
                        };

                        let argument = render_math_formula(&arguments[0]);

                        result.push(vec![
                            func_name,
                            vec![U16_LEFT_CURLY_BRACE],
                            argument,
                            vec![U16_RIGHT_CURLY_BRACE],
                        ].concat());

                        tmp = vec![];
                        index = indexes[0] + 1;
                    }

                    else {
                        result.push(tmp);
                        result.push(vec![content[index]]);
                        tmp = vec![];
                        index += 1;
                    }

                    continue;
                }

                else if LATEX_2ARG_FUNCS.contains(&tmp) {
                    let (arguments, indexes) = get_arguments(&content, index, vec![], vec![]);

                    if arguments.len() == 0 {
                        result.push(tmp);
                        result.push(vec![content[index]]);
                        tmp = vec![];
                        index += 1;
                    }

                    else if arguments.len() == 1 {

                        if tmp == into_v16("sqrt") {
                            let argument = render_math_formula(&arguments[0]);

                            result.push(vec![
                                into_v16("\\sqrt{"),
                                argument,
                                vec![U16_RIGHT_CURLY_BRACE],
                            ].concat());

                            tmp = vec![];
                            index = indexes[0] + 1;
                        }

                        else {
                            result.push(tmp);
                            result.push(vec![content[index]]);
                            tmp = vec![];
                            index += 1;
                        }

                    }

                    else if arguments.len() >= 2 {
                        let argument1 = render_math_formula(&arguments[0]);
                        let argument2 = render_math_formula(&arguments[1]);

                        if tmp == into_v16("frac") || tmp == into_v16("cfrac") {
                            result.push(vec![
                                vec![U16_BACKSLASH],
                                tmp,
                                vec![U16_LEFT_CURLY_BRACE],
                                argument1,
                                vec![U16_RIGHT_CURLY_BRACE, U16_LEFT_CURLY_BRACE],
                                argument2,
                                vec![U16_RIGHT_CURLY_BRACE]
                            ].concat());
                        }

                        else if tmp == into_v16("sqrt") {
                            result.push(vec![
                                vec![U16_BACKSLASH],
                                tmp,
                                vec![U16_LEFT_SQUARE_BRACKET],
                                argument1,
                                vec![U16_RIGHT_SQUARE_BRACKET, U16_LEFT_CURLY_BRACE],
                                argument2,
                                vec![U16_RIGHT_CURLY_BRACE]
                            ].concat());
                        }

                        else {
                            result.push(vec![
                                vec![U16_BACKSLASH],
                                tmp,
                                vec![U16_UNDERBAR, U16_LEFT_CURLY_BRACE],
                                argument1,
                                vec![U16_RIGHT_CURLY_BRACE, U16_CARET, U16_LEFT_CURLY_BRACE],
                                argument2,
                                vec![U16_RIGHT_CURLY_BRACE]
                            ].concat());
                        }

                        tmp = vec![];
                        index = indexes[1] + 1;
                    }

                    continue;
                }

                else {
                    result.push(tmp);
                }

                tmp = vec![];
            }

            result.push(vec![content[index]]);
        }

        index += 1;
    }

    result.pop();  // space character that I temporalily pushed at the beginning of this function
    result.concat()
}


fn get_arguments(content: &Vec<u16>, index: usize, mut current_args: Vec<Vec<u16>>, mut arg_end_indexes: Vec<usize>) -> (Vec<Vec<u16>>, Vec<usize>) {

    if index >= content.len() {
        (current_args, arg_end_indexes)
    }

    else if content[index] == U16_LEFT_CURLY_BRACE {

        match get_curly_brace_end_index(content, index) {
            None => (current_args, arg_end_indexes),
            Some(i) => {
                current_args.push(content[index + 1..i].to_vec());
                arg_end_indexes.push(i);

                get_arguments(content, i + 1, current_args, arg_end_indexes)
            }
        }

    }

    else if content[index] == U16_SPACE {
        get_arguments(content, index + 1, current_args, arg_end_indexes)
    }

    else {
        (current_args, arg_end_indexes)
    }

}


const LATEX_SYMBOLS_RAW: [&str;96] = ["leq", "geq", "ll", "gg", "equiv", "subset", "supset", "approx", "in", "ni", "subseteq", "supseteq", "cong", "simeq", "notin", "propto", "neq", "therefore", "because", "pm", "mp", "times", "div", "star", "cap", "cup", "vee", "wedge", "cdot", "diamond", "bullet", "oplus", "ominus", "otimes", "oslash", "odot", "circ", "exists", "nexists", "forall", "neg", "land", "lor", "rightarrow", "leftarrow", "iff", "top", "bot", "varnothing", "quad", "backslash", "leftcurlybrace", "rightcurlybrace", "alpha", "beta", "gamma", "Gamma", "delta", "Delta", "epsilon", "zeta", "eta", "theta", "Theta", "iota", "kappa", "lambda", "Lambda", "mu", "nu", "xi", "Xi", "pi", "Pi", "rho", "sigma", "Sigma", "tau", "upsilon", "Upsilon", "phi", "Phi", "chi", "psi", "Psi", "omega", "Omega", "partial", "nabla", "infty", "cos", "sin", "tan", "cosh", "sinh", "tanh"];
const LATEX_1ARG_FUNCS_RAW: [&str;12] = ["sub", "sup", "hat", "bar", "dot", "tilde", "vec", "check", "overleftarrow", "overrightarrow", "underline", "text"];
const LATEX_2ARG_FUNCS_RAW: [&str;9] = ["frac", "cfrac", "sqrt", "int", "oint", "iint", "iiint", "sum", "prod"];


#[cfg(test)]
mod tests {

    #[test]
    fn math_test1() {
        let orig = crate::utils::into_v16("sub{sub{-3}}sup {-x}{4} neq infty text{1234}");
        let rendered = String::from_utf16_lossy(&crate::markdown::math::render_math_formula(&orig));

        assert_eq!(rendered, "_{_{-3}}^{-x}{4} \\neq  \\infty  \\text{1234}".to_string());
    }

    #[test]
    fn math_test2() {
        let orig = crate::utils::into_v16("sqrt{1 + sqrt{2 + 3}} leq sqrt{3}{5 + frac   {2 + 5} {3 + 7}}");
        let rendered = String::from_utf16_lossy(&crate::markdown::math::render_math_formula(&orig));

        assert_eq!(rendered, "\\sqrt{1 + \\sqrt{2 + 3}} \\leq  \\sqrt[3]{5 + \\frac{2 + 5}{3 + 7}}".to_string());
    }

    #[test]
    fn math_test3() {
        let orig = crate::utils::into_v16("frac {3} sum {3} sqrt {3}");
        let rendered = String::from_utf16_lossy(&crate::markdown::math::render_math_formula(&orig));

        assert_eq!(rendered, "frac {3} sum {3} \\sqrt{3}".to_string());
    }

    #[test]
    fn math_test4() {
        let orig = crate::utils::into_v16("int{0}{infty} e sup{-x} dx");
        let rendered = String::from_utf16_lossy(&crate::markdown::math::render_math_formula(&orig));

        assert_eq!(rendered, "\\int_{0}^{\\infty } e ^{-x} dx".to_string());
    }

    #[test]
    fn math_test5() {
        let orig = crate::utils::into_v16("(vec{a} neq vec {b}) = (hat{a} neq hat {b})");
        let rendered = String::from_utf16_lossy(&crate::markdown::math::render_math_formula(&orig));

        assert_eq!(rendered, "(\\vec{a} \\neq  \\vec{b}) = (\\hat{a} \\neq  \\hat{b})".to_string());
    }

}