// This file is part of Hexchat Plugin API Bindings for Rust
// Copyright (C) 2018, 2021 Soni L.
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

use std::borrow::Cow;

/// A hexchat_get_info key.
#[derive(Eq, PartialEq, Ord, PartialOrd, Hash, Debug, Copy, Clone)]
pub enum InfoId<'a> {
    /// Returns the away message, or `None` if the user is not away.
    Away,
    /// Returns the current channel name.
    Channel,
    /// Returns the current charset.
    Charset,
    /// Returns the hexchat configuration directory, e.g. `/home/user/.config/hexchat`.
    Configdir,
    /// Returns the text event format string for the given text event name.
    EventText(&'a str),
    /// Returns the (real) hostname of the current server.
    Host,
    /// Returns the contents of the input box.
    Inputbox,
    // TODO replace with a get_libdirfs function!
    // /// Returns the library directory, e.g. `/usr/lib/hexchat`.
    // ///
    // /// May not always work, as this string isn't necessarily UTF-8, but local file system
    // /// encoding.
    // Libdirfs,
    /// Returns the channel modes, if known, or `None`.
    Modes,
    /// Returns the current network name, or `None`.
    Network,
    /// Returns the user's current nick.
    Nick,
    /// Returns the user's nickserv password, if any, or `None`
    Nickserv,
    /// Returns the current server name, or `None` if you are not connected.
    Server,
    /// Returns the current channel topic.
    Topic,
    /// Returns the HexChat version string.
    Version,
    /// Returns the window status: "active", "hidden" or "normal".
    WinStatus,
}

impl<'a> InfoId<'a> {
    pub fn name(&self) -> Cow<'static, str> {
        match *self {
            InfoId::EventText(s) => {
                let mut eventtext: String = "event_text ".into();
                eventtext.push_str(&s);
                eventtext.into()
            },
            InfoId::Away      => "away".into(),
            InfoId::Channel   => "channel".into(),
            InfoId::Charset   => "charset".into(),
            InfoId::Configdir => "configdir".into(),
            InfoId::Host      => "host".into(),
            InfoId::Inputbox  => "inputbox".into(),
            //InfoId::Libdirfs  => "libdirfs".into(),
            InfoId::Modes     => "modes".into(),
            InfoId::Network   => "network".into(),
            InfoId::Nick      => "nick".into(),
            InfoId::Nickserv  => "nickserv".into(),
            InfoId::Server    => "server".into(),
            InfoId::Topic     => "topic".into(),
            InfoId::Version   => "version".into(),
            InfoId::WinStatus => "win_status".into(),
        }
    }
}
