// This file is part of Hexchat Plugin API Bindings for Rust
// Copyright (C) 2022 Soni L.
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

/// Stripping mode for [`PluginHandle::strip`](crate::PluginHandle::strip).
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash, Default)]
pub struct Strip {
    // 4
    hidden: bool,
    // 2
    formatting: bool,
    // 1
    colors: bool,
}

impl Strip {
    /// Creates a new `Strip` that, by default, strips no attributes.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// use hexchat_unsafe_plugin::{PluginHandle, Strip};
    ///
    /// fn strip_nothing(s: &str, ph: &PluginHandle<'_>) -> String {
    ///     ph.strip(s, Strip::new())
    /// }
    /// ```
    #[inline]
    pub const fn new() -> Strip {
        Strip {
            hidden: false,
            formatting: false,
            colors: false,
        }
    }

    /// Sets whether to remove mIRC color attributes.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// use hexchat_unsafe_plugin::{PluginHandle, Strip};
    ///
    /// fn strip_colors(s: &str, ph: &PluginHandle<'_>) -> String {
    ///     ph.strip(s, Strip::new().colors(true))
    /// }
    /// ```
    #[inline]
    pub const fn colors(mut self, strip: bool) -> Self {
        self.colors = strip;
        self
    }

    /// Sets whether to remove formatting attributes.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// use hexchat_unsafe_plugin::{PluginHandle, Strip};
    ///
    /// fn strip_formatting(s: &str, ph: &PluginHandle<'_>) -> String {
    ///     ph.strip(s, Strip::new().formatting(true))
    /// }
    /// ```
    #[inline]
    pub const fn formatting(mut self, strip: bool) -> Self {
        self.formatting = strip;
        self
    }

    /// Sets whether to remove internal "hidden text" formatting attributes.
    ///
    /// This is split from [`Self::formatting`] because these attributes are
    /// only processed when writing directly to a buffer - they're for
    /// internal/plugin use. This tends to be useful when processing user or
    /// remote input and writing it directly to a buffer.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// use hexchat_unsafe_plugin::{PluginHandle, Strip};
    ///
    /// fn strip_hidden(s: &str, ph: &PluginHandle<'_>) -> String {
    ///     ph.strip(s, Strip::new().hidden(true))
    /// }
    /// ```
    #[inline]
    pub const fn hidden(mut self, strip: bool) -> Self {
        self.hidden = strip;
        self
    }

    /// Creates a new `Strip` that strips all strippable attributes.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// use hexchat_unsafe_plugin::{PluginHandle, Strip};
    ///
    /// fn strip_all(s: &str, ph: &PluginHandle<'_>) -> String {
    ///     ph.strip(s, Strip::all())
    /// }
    /// ```
    #[inline]
    pub const fn all() -> Strip {
        Strip {
            hidden: true,
            formatting: true,
            colors: true,
        }
    }

    /// Builds the flags for FFI.
    pub(crate) fn flags(self) -> ::libc::c_int {
        let mut value = 0;
        if self.hidden {
            value |= 4;
        }
        if self.formatting {
            value |= 2;
        }
        if self.colors {
            value |= 1;
        }
        value
    }
}
