use core::mem::MaybeUninit;
use core::slice;

use crate::{AsciiCase, Error};

fn ok_cases() -> Vec<Vec<u8>> {
    let mut ans = Vec::new();

    for n in 0..256usize {
        let iter = (0..16)
            .cycle()
            .take(n)
            .map(|x| char::from_digit(x, 16).unwrap() as u8);
        ans.push(iter.collect())
    }

    ans
}

fn err_cases() -> Vec<Vec<u8>> {
    vec![
        vec![0],
        vec![b'0', 0],
        vec![b'a', b'f', 0],
        vec![b'a', b'0', b'c', 0],
        vec![b'a', b'0', b'c', b'1', 0],
    ]
}

fn as_mut<T>(s: &mut [T]) -> &mut [MaybeUninit<T>] {
    let (ptr, len) = (s.as_mut_ptr(), s.len());
    unsafe { slice::from_raw_parts_mut(ptr.cast(), len) }
}

pub fn test(
    check: impl Fn(&[u8]) -> bool,
    decode: impl for<'s, 'd> Fn(&'s [u8], &'d mut [MaybeUninit<u8>]) -> Result<&'d mut [u8], Error>,
    encode: impl for<'s, 'd> Fn(
        &'s [u8],
        &'d mut [MaybeUninit<u8>],
        AsciiCase,
    ) -> Result<&'d mut [u8], Error>,
) {
    let ok_cases = ok_cases();
    let err_cases = err_cases();

    for src in &ok_cases {
        assert!(check(src));
        if src.len() % 2 == 0 {
            let mut decode_buf = vec![0; src.len() / 2];
            let mut encode_buf = vec![0; src.len()];

            let decode_buf = decode(src, as_mut(&mut decode_buf)).unwrap();
            let encode_buf = encode(decode_buf, as_mut(&mut encode_buf), AsciiCase::Lower).unwrap();

            assert_eq!(encode_buf, src);
        } else {
            let mut encode_buf = vec![0; src.len() * 2];
            let mut decode_buf = vec![0; src.len()];
            let encode_buf = encode(src, as_mut(&mut encode_buf), AsciiCase::Upper).unwrap();
            dbg!(src, core::str::from_utf8(encode_buf).unwrap());
            let decode_buf = decode(encode_buf, as_mut(&mut decode_buf)).unwrap();

            assert_eq!(decode_buf, src);
        }
    }

    for src in &err_cases {
        assert!(!check(src));
        let mut buf = vec![0; src.len() / 2];
        assert!(decode(src, as_mut(&mut buf)).is_err())
    }
}
