use core::mem::MaybeUninit;

use crate::{AsciiCase, Error};

macro_rules! try_simd {
    ($f:ident($($args:tt)*)) => {
        #[cfg(any(target_arch = "x86", target_arch = "x86_64"))]
        {
            use simd_abstraction::InstructionSet;
            use simd_abstraction::x86::*;
            if AVX2::detect().is_some() {
                return unsafe { $crate::avx2::$f($($args)*) };
            }
            if SSE41::detect().is_some() {
                return unsafe { $crate::sse41::$f($($args)*) };
            }
        }
    };
}

#[inline]
pub fn check(src: &[u8]) -> bool {
    try_simd!(check(src));
    crate::fallback::check(src)
}

#[inline]
pub fn encode<'s, 'd>(
    src: &'s [u8],
    dst: &'d mut [MaybeUninit<u8>],
    case: AsciiCase,
) -> Result<&'d mut [u8], Error> {
    try_simd!(encode(src, dst, case));
    crate::fallback::encode(src, dst, case)
}

#[inline]
pub fn decode<'s, 'd>(
    src: &'s [u8],
    dst: &'d mut [MaybeUninit<u8>],
) -> Result<&'d mut [u8], Error> {
    try_simd!(decode(src, dst));
    crate::fallback::decode(src, dst)
}

#[cfg(feature = "alloc")]
use alloc::boxed::Box;

#[cfg(feature = "alloc")]
#[inline]
pub fn encode_boxed_str(src: &[u8], case: AsciiCase) -> Box<str> {
    use alloc::alloc::{alloc, handle_alloc_error, Layout};
    use core::slice;
    use core::str;

    #[inline]
    unsafe fn alloc_uninit_bytes(len: usize) -> alloc::boxed::Box<[MaybeUninit<u8>]> {
        let layout = Layout::from_size_align_unchecked(len, 1);
        let p = alloc(layout);
        if p.is_null() {
            handle_alloc_error(layout)
        }
        let ptr = p.cast();
        Box::from_raw(slice::from_raw_parts_mut(ptr, len))
    }

    if src.is_empty() {
        return alloc::string::String::new().into_boxed_str();
    }

    unsafe {
        let mut uninit_buf = alloc_uninit_bytes(src.len() * 2);
        encode(src, &mut uninit_buf, case).unwrap();

        let len = uninit_buf.len();
        let ptr = Box::into_raw(uninit_buf).cast::<u8>();
        let buf = slice::from_raw_parts_mut(ptr, len);
        Box::from_raw(str::from_utf8_unchecked_mut(buf))
    }
}

#[test]
fn test_str() {
    let ans = encode_boxed_str("hello".as_ref(), AsciiCase::Lower);
    assert_eq!(&*ans, "68656c6c6f");
}
