use std::{
    collections::{BTreeMap, BTreeSet},
    fmt::Write,
    sync::atomic::Ordering,
};

use atomic_traits::{Atomic, NumOps};
use num_traits::{One, Signed, Unsigned, Zero};
use parking_lot::{lock_api::RwLockUpgradableReadGuard, RwLock};

use crate::{Label, Metric, NameError, Value};

/// Counter metric
///
/// From the Prometheus docs:
///
/// > A counter is a cumulative metric that represents a single monotonically
/// > increasing counter whose value can only increase or be reset to zero on
/// > restart. For example, you can use a counter to represent the number of
/// > requests served, tasks completed, or errors.
///
/// > Do not use a counter to expose a value that can decrease. For example, do
/// > not use a counter for the number of currently running processes; instead
/// > use a gauge.
///
/// # Examples
///
/// ```
/// use std::sync::atomic::{Ordering, AtomicUsize};
/// use hesione::{labels, Counter, Metric};
///
/// // Create an example counter
/// let counter = Counter::<AtomicUsize>::new("example").unwrap();
///
/// // Increment the counter a few times
/// counter.inc(Ordering::SeqCst, None);
/// counter.inc(Ordering::SeqCst, None);
/// counter.inc(Ordering::SeqCst, None);
///
/// // When we convert the metric into Prometheus' format, it'll look like this:
/// let expected = concat!(
///     "# TYPE example counter\n",
///     "example 3\n",
///     "\n",
/// );
///
/// let mut actual = String::with_capacity(64);
/// counter.to_prometheus_lines(&mut actual, Ordering::SeqCst);
///
/// assert_eq!(actual, expected);
/// ```
pub struct Counter<T> {
    // Name of the metric
    name: String,

    // Longer description of what metric measures
    help: Option<String>,

    // Little explanation of this data structure:
    //
    // * (None, T) is `metric_name 1337`
    // * (Some(BTreeSet<Label>), T) would be something like
    //   `metric_name{foo="bar", baz="qux"} 1337`
    values: RwLock<BTreeMap<Option<BTreeSet<Label>>, T>>,
}

impl<T> Counter<T> {
    /// Create a new [`Counter`][Counter]-type metric
    pub fn new<S: ToString>(name: S) -> Result<Self, NameError> {
        let name = name.to_string();

        if crate::is_valid_name(&name) {
            Ok(Self {
                name,
                help: Default::default(),
                values: Default::default(),
            })
        } else {
            Err(NameError)
        }
    }

    /// Set the help text for this counter
    pub fn help<S: ToString>(mut self, help: S) -> Self {
        self.help = Some(help.to_string());

        self
    }
}

impl<T> Counter<T>
where
    T: Default + Atomic,
{
    /// Gets the current value of the counter
    pub fn get(
        &self,
        order: Ordering,
        labels: Option<BTreeSet<Label>>,
    ) -> <T as Atomic>::Type {
        let values = self.values.upgradable_read();

        match values.get(&labels) {
            Some(x) => x.load(order),
            None => {
                let mut values = RwLockUpgradableReadGuard::upgrade(values);
                let x = values.entry(labels).or_default();

                x.load(order)
            }
        }
    }
}

impl<T> Counter<T>
where
    T: Default + NumOps + Atomic,
    <T as Atomic>::Type: One,
{
    /// Increments the counter by 1
    pub fn inc(&self, order: Ordering, labels: Option<BTreeSet<Label>>) {
        self.add_unchecked(order, labels, <T as Atomic>::Type::one());
    }
}

impl<T> Counter<T>
where
    T: Default + NumOps + Atomic,
{
    fn add_unchecked(
        &self,
        order: Ordering,
        labels: Option<BTreeSet<Label>>,
        add: <T as Atomic>::Type,
    ) {
        let values = self.values.upgradable_read();

        match values.get(&labels) {
            Some(x) => {
                x.fetch_add(add, order);
            }
            None => {
                let mut values = RwLockUpgradableReadGuard::upgrade(values);
                let x = values.entry(labels).or_default();

                x.fetch_add(add, order);
            }
        }
    }
}

impl<T> Counter<T>
where
    T: Default + NumOps + Atomic,
    <T as Atomic>::Type: One + Zero + Unsigned,
{
    /// Increment the counter by an arbitrary unsigned value
    ///
    /// This doesn't do any runtime assertion that `add` is zero or positive
    /// since that invariant is proven at compile time with the
    /// [`Unsigned`](num_traits::Unsigned) bound.
    pub fn add(
        &self,
        order: Ordering,
        labels: Option<BTreeSet<Label>>,
        add: <T as Atomic>::Type,
    ) {
        self.add_unchecked(order, labels, add);
    }
}

impl<T> Counter<T>
where
    T: Default + NumOps + Atomic,
    <T as Atomic>::Type: One + Zero + Signed,
{
    /// Increment the counter by an arbitrary signed value
    ///
    /// # Panics
    ///
    /// Panics if `add` is less than zero.
    pub fn add_signed(
        &self,
        order: Ordering,
        labels: Option<BTreeSet<Label>>,
        add: <T as Atomic>::Type,
    ) {
        if add.is_negative() {
            panic!(
                "counters cannot decrease, perhaps you want a gauge instead"
            );
        }

        self.add_unchecked(order, labels, add);
    }
}

impl<T, W> Metric<W> for Counter<T>
where
    T: Value,
    W: Write,
{
    fn to_prometheus_lines(&self, buf: &mut W, order: Ordering) {
        if let Some(help) = &self.help {
            writeln!(buf, "# HELP {} {}", self.name, help).unwrap();
        }

        writeln!(buf, "# TYPE {} counter", self.name).unwrap();

        for (k, v) in self.values.read().iter() {
            let text = v.to_prometheus_string(order);

            if let Some(labels) = k {
                write!(buf, "{}", self.name).unwrap();

                if !labels.is_empty() {
                    write!(buf, "{{").unwrap();
                }

                let len = labels.len();
                for (i, label) in labels.iter().enumerate() {
                    let maybe_separator = if i + 1 == len {
                        // This is the end
                        ""
                    } else {
                        // We need another separator
                        ", "
                    };
                    write!(
                        buf,
                        "{}=\"{}\"{}",
                        label.name, label.value, maybe_separator
                    )
                    .unwrap();
                }

                if !labels.is_empty() {
                    write!(buf, "}}").unwrap();
                }

                writeln!(buf, " {}", text).unwrap();
            } else {
                writeln!(buf, "{} {}", self.name, text).unwrap();
            }
        }

        // Just for readability in the output
        writeln!(buf).unwrap();
    }
}

#[cfg(test)]
mod tests {
    #[test]
    fn labelled_increment() {
        use std::sync::atomic::{AtomicU8, Ordering};

        use crate::{labels, Counter};

        let counter = Counter::<AtomicU8>::new("tests").unwrap();

        let labels = labels! {
            "foo": "bar".to_string(),
        }
        .unwrap();

        counter.inc(Ordering::SeqCst, Some(labels.clone()));
        counter.inc(Ordering::SeqCst, Some(labels.clone()));

        let x = counter.get(Ordering::SeqCst, Some(labels));
        assert_eq!(x, 2);
    }

    #[test]
    fn no_values_to_lines() {
        use std::sync::atomic::{AtomicU8, Ordering};

        use crate::{Counter, Metric};

        let counter = Counter::<AtomicU8>::new("tests")
            .unwrap()
            .help("The number of times this test has been run");

        let expected = concat!(
            "# HELP tests The number of times this test has been run\n",
            "# TYPE tests counter\n",
            "\n",
        );

        let mut buf = String::with_capacity(128);
        counter.to_prometheus_lines(&mut buf, Ordering::SeqCst);
        assert_eq!(expected, buf);
    }

    #[test]
    fn one_value_no_labels_to_lines() {
        use std::sync::atomic::{AtomicU8, Ordering};

        use crate::{Counter, Metric};

        let counter = Counter::<AtomicU8>::new("tests")
            .unwrap()
            .help("The number of times this test has been run");

        counter.inc(Ordering::SeqCst, None);
        counter.inc(Ordering::SeqCst, None);

        let expected = concat!(
            "# HELP tests The number of times this test has been run\n",
            "# TYPE tests counter\n",
            "tests 2\n",
            "\n",
        );

        let mut buf = String::with_capacity(128);
        counter.to_prometheus_lines(&mut buf, Ordering::SeqCst);
        assert_eq!(expected, buf);
    }

    #[test]
    fn one_value_one_label_to_lines() {
        use std::sync::atomic::{AtomicU8, Ordering};

        use crate::{labels, Counter, Metric};

        let counter = Counter::<AtomicU8>::new("tests")
            .unwrap()
            .help("The number of times this test has been run");

        let labels = labels! {
            "foo": "bar".to_string(),
        }
        .unwrap();

        counter.inc(Ordering::SeqCst, Some(labels.clone()));
        counter.inc(Ordering::SeqCst, Some(labels));

        let expected = concat!(
            "# HELP tests The number of times this test has been run\n",
            "# TYPE tests counter\n",
            "tests{foo=\"bar\"} 2\n",
            "\n",
        );

        let mut buf = String::with_capacity(128);
        counter.to_prometheus_lines(&mut buf, Ordering::SeqCst);
        assert_eq!(expected, buf);
    }

    #[test]
    fn one_value_two_labels_to_lines() {
        use std::sync::atomic::{AtomicU8, Ordering};

        use crate::{labels, Counter, Metric};

        let counter = Counter::<AtomicU8>::new("tests")
            .unwrap()
            .help("The number of times this test has been run");

        let labels = labels! {
            "foo": "bar".to_string(),
            "baz": "qux".to_string(),
        }
        .unwrap();

        counter.inc(Ordering::SeqCst, Some(labels.clone()));
        counter.inc(Ordering::SeqCst, Some(labels));

        let expected = concat!(
            "# HELP tests The number of times this test has been run\n",
            "# TYPE tests counter\n",
            // This order is deterministic because it's a BTreeSet and not a
            // HashSet
            "tests{baz=\"qux\", foo=\"bar\"} 2\n",
            "\n",
        );

        let mut buf = String::with_capacity(128);
        counter.to_prometheus_lines(&mut buf, Ordering::SeqCst);
        assert_eq!(expected, buf);
    }

    #[test]
    fn add() {
        use std::sync::atomic::{AtomicU8, Ordering};

        use crate::Counter;

        let counter = Counter::<AtomicU8>::new("tests")
            .unwrap()
            .help("The number of times this test has been run");

        let expected = 10;

        counter.add(Ordering::SeqCst, None, expected);

        assert_eq!(expected, counter.get(Ordering::SeqCst, None));
    }

    #[test]
    #[should_panic]
    fn add_signed() {
        use std::sync::atomic::{AtomicI8, Ordering};

        use crate::Counter;

        let counter = Counter::<AtomicI8>::new("tests")
            .unwrap()
            .help("The number of times this test has been run");

        counter.add_signed(Ordering::SeqCst, None, -10);
    }
}
