/// A label that can be applied to metrics
///
/// For example, with the following label applied to a counter metric named
/// `requests`,
///
/// ```
/// # use hesione::Label;
/// let l = Label {
///     name: "response_code".to_string(),
///     value: "404".to_string(),
/// };
/// ```
///
/// The output Prometheus would see will look like this:
///
/// ```text
/// requests{response_code="404"} 1337
/// ```
#[derive(PartialOrd, Ord, Hash, PartialEq, Eq, Clone, Debug)]
pub struct Label {
    /// Name of the label
    pub name: String,

    /// The value of the label
    pub value: String,
}

/// Easily generate metric labels
///
/// ```
/// # use hesione::{Label, labels};
/// # let addr = "placeholder";
/// let labels = labels! {
///     "response_code": "404".to_string(),
///     "client_ip": addr.to_string(),
/// }.unwrap();
/// ```
///
/// This will generate a `Result<BTreeMap<Label>, NameError>` where the label
/// name is the string literal on the left of the `:` and the value is the
/// evaluated expression on the right.
#[macro_export]
macro_rules! labels {
    ( $($name:literal: $value:expr),+ $(,)? ) => ({
        use ::std::collections::BTreeSet;

        let mut labels = BTreeSet::new();

        let f = || {
            $(
                let name = $name.to_string();

                if !$crate::is_valid_name(&name) {
                    return Err($crate::NameError);
                } else {
                    labels.insert($crate::Label {
                        name,
                        value: $value,
                    });
                }
            )+

            Ok(labels)
        };

        f()
    });
}
