use std::collections::HashMap;
use std::iter::Iterator;
use std::hash::Hash;
use crate::state::State;

struct StateMachine<StateId: Eq + Hash + Copy, Input> {
    states: HashMap<StateId, State<StateId, Input>>,
    current_state: StateId,
}

impl<StateId: Eq + Hash + Copy, Input> StateMachine<StateId, Input> {
    fn new(initial_state: StateId, states: impl IntoIterator<Item=State<StateId, Input>>) -> Self {
        Self {
            states: states.into_iter().map(|state| (state.id(), state)).collect(),
            current_state: initial_state,
        }
    }

    fn current_state(&self) -> StateId {
        self.current_state
    }

    fn handle_input(&mut self, input: &Input) -> Option<StateId> {
        let current_state = self.states.get_mut(&self.current_state)?;
        self.current_state = current_state.handle_input(input)?;
        Some(self.current_state)
    }
}

#[cfg(test)]
mod tests {
    use crate::state::State;
    use crate::state_machine::StateMachine;
    use crate::handler::*;
    use crate::state_machine::tests::LockInput::CorrectKey;
    use crate::state_machine::tests::LockState::{Locked, Unlocked};

    #[derive(PartialEq, Eq, Hash, Clone, Copy, Debug)]
    enum LockState {
        Locked,
        Unlocked,
    }

    #[derive(PartialEq, Eq, Hash, Clone, Copy, Debug)]
    enum LockInput {
        CorrectKey,
        IncorrectKey,
    }

    #[test]
    fn simple_lock() {
        let mut machine = StateMachine::new(Locked, vec![
            State::new(Locked, Box::new(StatelessHandler::new(
                |input: &LockInput| -> Option<LockState> {
                    Some(match input {
                        LockInput::CorrectKey => LockState::Unlocked,
                        LockInput::IncorrectKey => LockState::Locked
                    })
                }
            ))),
            State::new(Unlocked, Box::new(StatelessHandler::new(
                |input: &LockInput| -> Option<LockState> {
                    Some(match input {
                        LockInput::CorrectKey => LockState::Locked,
                        LockInput::IncorrectKey => LockState::Unlocked
                    })
                }
            )))
        ]);

        assert_eq!(machine.current_state(), LockState::Locked);
        machine.handle_input(&LockInput::CorrectKey);
        assert_eq!(machine.current_state(), LockState::Unlocked);
        machine.handle_input(&LockInput::CorrectKey);
        assert_eq!(machine.current_state(), LockState::Locked);
    }

    #[derive(PartialEq, Eq, Hash, Clone, Copy, Debug)]
    enum SecureLockState {
        Locked,
        Unlocked,
        Terminal,
    }

    struct LockedContext {
        unlock_attempts: usize,
        max_failed_attempts: usize
    }

    impl LockedContext {
        fn new(max_failed_attempts: usize) -> Self {
            Self { unlock_attempts: 0, max_failed_attempts }
        }
    }

    #[test]
    fn secure_lock() {
        const MAX_ATTEMPTS: usize = 3;

        let mut machine = StateMachine::new(SecureLockState::Locked, vec![
            State::new(SecureLockState::Locked,
                       Box::new(StatefulHandler::new(
                           LockedContext::new(MAX_ATTEMPTS),
                           |context: &mut LockedContext, input: &LockInput| -> Option<SecureLockState> {
                               match input {
                                   LockInput::CorrectKey => {
                                       context.unlock_attempts = 0;
                                       Some(SecureLockState::Unlocked)
                                   },
                                   LockInput::IncorrectKey => {
                                       context.unlock_attempts += 1;

                                       if context.unlock_attempts >= context.max_failed_attempts {
                                           return Some(SecureLockState::Terminal)
                                       }

                                       Some(SecureLockState::Locked)
                                   }
                               }
                           }))),
            State::new(SecureLockState::Unlocked, Box::new(StatelessHandler::new(
                |input: &LockInput| -> Option<SecureLockState> {
                    Some(match input {
                        LockInput::CorrectKey => SecureLockState::Locked,
                        LockInput::IncorrectKey => SecureLockState::Unlocked
                    })
                }
            ))),
            State::new(SecureLockState::Terminal, Box::new(StatelessHandler::new(
                |input: &LockInput| -> Option<SecureLockState> {
                    Some(SecureLockState::Terminal)
                }
            )))
        ]);

        machine.handle_input(&LockInput::IncorrectKey);
        assert_eq!(machine.current_state(), SecureLockState::Locked);
        machine.handle_input(&LockInput::CorrectKey);
        assert_eq!(machine.current_state(), SecureLockState::Unlocked);
        machine.handle_input(&LockInput::CorrectKey);
        assert_eq!(machine.current_state(), SecureLockState::Locked);

        for _ in 0..MAX_ATTEMPTS - 1 {
            machine.handle_input(&LockInput::IncorrectKey);
            assert_eq!(machine.current_state(), SecureLockState::Locked);
        }

        machine.handle_input(&LockInput::IncorrectKey);
        assert_eq!(machine.current_state(), SecureLockState::Terminal);
    }
}
