use std::marker::PhantomData;

pub trait Handler<StateId, Input> {
    fn handle_input(&mut self, input: &Input) -> Option<StateId>;
}

pub struct StatelessHandler<StateId, Input, HandlingFunction>
    where HandlingFunction: Fn(&Input) -> Option<StateId> {
    state_id_type: PhantomData<StateId>,
    state_input_type: PhantomData<Input>,
    handling_function: HandlingFunction,
}

impl<StateId, Input, HandlingFunction> StatelessHandler<StateId, Input, HandlingFunction>
    where HandlingFunction: Fn(&Input) -> Option<StateId> {
    pub fn new(handling_function: HandlingFunction) -> Self {
        Self { state_id_type: PhantomData, state_input_type: PhantomData, handling_function }
    }
}

impl<StateId, Input, HandlingFunction> Handler<StateId, Input> for StatelessHandler<StateId, Input, HandlingFunction>
    where HandlingFunction: Fn(&Input) -> Option<StateId> {
    fn handle_input(&mut self, input: &Input) -> Option<StateId> {
        (self.handling_function)(input)
    }
}

pub struct StatefulHandler<StateId, Input, Context, HandlingFunction>
    where HandlingFunction: Fn(&mut Context, &Input) -> Option<StateId> {
    state_id_type: PhantomData<StateId>,
    state_input_type: PhantomData<Input>,
    state_context_type: PhantomData<Context>,
    context: Context,
    handling_function: HandlingFunction,
}

impl<StateId, Input, Context, HandlingFunction> StatefulHandler<StateId, Input, Context, HandlingFunction>
    where HandlingFunction: Fn(&mut Context, &Input) -> Option<StateId> {
    pub fn new(context: Context, handling_function: HandlingFunction) -> Self {
        Self {
            state_id_type: PhantomData,
            state_input_type: PhantomData,
            state_context_type: PhantomData,
            context,
            handling_function
        }
    }
}

impl<StateId, Input, Context, HandlingFunction> Handler<StateId, Input> for StatefulHandler<StateId, Input, Context, HandlingFunction>
    where HandlingFunction: Fn(&mut Context, &Input) -> Option<StateId> {
    fn handle_input(&mut self, input: &Input) -> Option<StateId> {
        (self.handling_function)(&mut self.context, input)
    }
}